<?php

namespace Drupal\comfyui\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'comfyui_text_image' formatter.
 *
 * @FieldFormatter(
 *   id = "comfyui_text_image",
 *   label = @Translation("ComfyUI Image (from text URL)"),
 *   field_types = {
 *     "string",
 *     "string_long",
 *     "text",
 *     "text_long"
 *   }
 * )
 */
class ComfyUITextImageFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'image_style' => '',
      'max_width' => '',
      'max_height' => '',
      'show_url' => FALSE,
      'alt_text' => 'ComfyUI Generated Image',
      'fallback_text' => 'No image available',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['image_style'] = [
      '#title' => $this->t('Image style'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('image_style'),
      '#empty_option' => $this->t('None (original image)'),
      '#options' => image_style_options(FALSE),
      '#description' => $this->t('Select an image style to apply to the ComfyUI images.'),
    ];

    $elements['max_width'] = [
      '#title' => $this->t('Max width (px)'),
      '#type' => 'number',
      '#default_value' => $this->getSetting('max_width'),
      '#min' => 0,
      '#description' => $this->t('Maximum width in pixels. Leave empty for no limit.'),
    ];

    $elements['max_height'] = [
      '#title' => $this->t('Max height (px)'),
      '#type' => 'number',
      '#default_value' => $this->getSetting('max_height'),
      '#min' => 0,
      '#description' => $this->t('Maximum height in pixels. Leave empty for no limit.'),
    ];

    $elements['alt_text'] = [
      '#title' => $this->t('Alt text'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('alt_text'),
      '#description' => $this->t('Alternative text for the image for accessibility.'),
    ];

    $elements['show_url'] = [
      '#title' => $this->t('Show URL below image'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('show_url'),
      '#description' => $this->t('Display the image URL below the image.'),
    ];

    $elements['fallback_text'] = [
      '#title' => $this->t('Fallback text'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('fallback_text'),
      '#description' => $this->t('Text to display when no valid image URL is found.'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    
    $image_style = $this->getSetting('image_style');
    if ($image_style) {
      $summary[] = $this->t('Image style: @style', ['@style' => $image_style]);
    } else {
      $summary[] = $this->t('Original image size');
    }

    $max_width = $this->getSetting('max_width');
    $max_height = $this->getSetting('max_height');
    if ($max_width || $max_height) {
      $summary[] = $this->t('Max dimensions: @width x @height', [
        '@width' => $max_width ?: 'auto',
        '@height' => $max_height ?: 'auto',
      ]);
    }

    if ($this->getSetting('show_url')) {
      $summary[] = $this->t('Show URL below image');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $url = $item->value;
      
      if (empty($url)) {
        continue;
      }

      // Check if this looks like a ComfyUI proxy URL or valid image URL
      if ($this->isValidImageUrl($url)) {
        // Build image element
        $image_element = [
          '#theme' => 'image',
          '#uri' => $url,
          '#alt' => $this->getSetting('alt_text'),
          '#title' => $this->getSetting('alt_text'),
          '#attributes' => [
            'class' => ['comfyui-text-image'],
          ],
        ];

        // Apply image style if configured
        $image_style = $this->getSetting('image_style');
        if (!empty($image_style)) {
          $image_element['#theme'] = 'image_style';
          $image_element['#style_name'] = $image_style;
        }

        // Apply max dimensions via inline styles
        $style_parts = [];
        $max_width = $this->getSetting('max_width');
        $max_height = $this->getSetting('max_height');
        
        if ($max_width) {
          $style_parts[] = "max-width: {$max_width}px";
        }
        if ($max_height) {
          $style_parts[] = "max-height: {$max_height}px";
        }
        
        if (!empty($style_parts)) {
          $image_element['#attributes']['style'] = implode('; ', $style_parts);
        }

        // Build the complete element
        $elements[$delta] = [
          '#type' => 'container',
          '#attributes' => [
            'class' => ['comfyui-text-image-container'],
          ],
          'image' => $image_element,
        ];

        // Add URL below image if configured
        if ($this->getSetting('show_url')) {
          $elements[$delta]['url'] = [
            '#markup' => '<div class="comfyui-image-url"><small>' . htmlspecialchars($url) . '</small></div>',
          ];
        }

      } else {
        // Not a valid image URL, show fallback text
        $elements[$delta] = [
          '#markup' => '<div class="comfyui-no-image">' . $this->getSetting('fallback_text') . '</div>',
        ];
      }
    }

    return $elements;
  }

  /**
   * Check if URL looks like a valid image URL.
   */
  protected function isValidImageUrl($url) {
    // Check for ComfyUI proxy URLs
    if (strpos($url, '/comfyui/image-proxy/') !== FALSE) {
      return TRUE;
    }
    
    // Check for common image extensions
    $image_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
    $extension = strtolower(pathinfo($url, PATHINFO_EXTENSION));
    
    return in_array($extension, $image_extensions);
  }

}
