<?php

namespace Drupal\comfyui\Plugin\QueueWorker;

use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\RequeueException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\comfyui\Service\ComfyUIWorkflowParser;
use RuntimeException;

/**
 * Process ComfyUI workflow executions.
 *
 * @QueueWorker(
 *   id = "comfyui_workflow_executor",
 *   title = @Translation("ComfyUI Workflow Executor"),
 *   cron = {"time" = 60}
 * )
 */
class ComfyUIWorkflowExecutor extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The workflow parser service.
   *
   * @var \Drupal\comfyui\Service\ComfyUIWorkflowParser
   */
  protected $workflowParser;

  /**
   * Maximum number of retry attempts.
   *
   * @var int
   */
  protected const MAX_RETRIES = 3;

  /**
   * Constructs a ComfyUIWorkflowExecutor object.
   *
   * @param array $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   Plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   Logger factory.
   * @param \Drupal\comfyui\Service\ComfyUIWorkflowParser $workflow_parser
   *   Workflow parser service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    ComfyUIWorkflowParser $workflow_parser
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->workflowParser = $workflow_parser;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('http_client'),
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('comfyui.workflow_parser')
    );
  }

   /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    try {
      // Initialize attempt counter if not set
      $data['attempts'] = $data['attempts'] ?? 0;
      $view_mode = $data['view_mode'] ?? 'form'; // Support view mode

      $workflow = $this->entityTypeManager
        ->getStorage('comfyui_workflow')
        ->load($data['workflow_id']);

      if (!$workflow) {
        throw new RuntimeException('Workflow not found');
      }

      $config = $this->configFactory->get('comfyui.settings');
      $api_endpoint = $config->get('api_endpoint');
      $timeout = $config->get('timeout');

      // Get and parse workflow JSON
      $workflow_json = $this->workflowParser->parseWorkflowJson(
        $workflow->getWorkflowApi()
      );

      // Apply field mappings using configuration
      $workflow_json = $this->applyViewModeMappings($workflow_json, $data['field_values'], $workflow, $view_mode);

      // Build request options
      $request_options = [
        'json' => $workflow_json,
        'timeout' => $timeout,
      ];
      
      // Add authentication if login module is enabled
      if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
        $token_service = \Drupal::service('comfyui_login.token_service');
        $request_options = $token_service->buildRequestOptions($request_options);
      }

      // Execute workflow
      $response = $this->httpClient->post($api_endpoint . '/execute', $request_options);

      $result = json_decode($response->getBody(), TRUE);

      if (json_last_error() !== JSON_ERROR_NONE) {
        throw new RuntimeException('Invalid JSON response from API');
      }

      // Process result
      $this->processWorkflowResult($workflow, $result, $data);

    }
    catch (RequestException $e) {
      // Handle network/API errors with retry logic
      if ($data['attempts'] < self::MAX_RETRIES) {
        $data['attempts']++;
        throw new RequeueException('Temporary failure, retrying... Attempt ' . $data['attempts']);
      }
      $this->handleExecutionFailure($workflow ?? NULL, $e);
      throw $e;
    }
    catch (\Exception $e) {
      $this->handleExecutionFailure($workflow ?? NULL, $e);
      throw $e;
    }
  }

  /**
   * Applies view mode field mappings to workflow JSON.
   *
   * @param array $workflow_json
   *   The workflow JSON data.
   * @param array $field_values
   *   The field values to apply.
   * @param object $workflow
   *   The workflow entity.
   * @param string $view_mode
   *   The view mode (not used since mappings are shared).
   *
   * @return array
   *   The modified workflow JSON.
   */
  protected function applyViewModeMappings(array $workflow_json, array $field_values, $workflow, $view_mode): array {
    // Get field mappings (shared across all view modes)
    $field_mappings = $workflow->getFieldMappings();
    
    foreach ($field_values as $field_name => $value) {
      $mapping = $field_mappings[$field_name] ?? NULL;
      if (!$mapping || $mapping['direction'] !== 'input') {
        continue;
      }

      $node_id = $mapping['comfyui_node'];
      $input_name = $mapping['comfyui_input'];
      
      if (isset($workflow_json[$node_id])) {
        $workflow_json[$node_id]['inputs'][$input_name] = $value;
      }
    }

    return $workflow_json;
  }

  /**
   * Handles workflow execution failures.
   *
   * @param mixed $workflow
   *   The workflow entity or NULL if not loaded.
   * @param \Exception $e
   *   The exception that occurred.
   */
  protected function handleExecutionFailure($workflow, \Exception $e): void {
    $context = [
      '@error' => $e->getMessage(),
      '@workflow_id' => $workflow ? $workflow->id() : 'unknown',
    ];

    $this->loggerFactory->get('comfyui')
      ->error('Workflow execution failed for workflow @workflow_id: @error', $context);

    if ($workflow) {
      // Update workflow status
      $workflow->set('status', FALSE)->save();
    }
  }

  /**
   * Applies field mappings to workflow JSON.
   *
   * @param array $workflow_json
   *   The workflow JSON data.
   * @param array $field_values
   *   The field values to apply.
   *
   * @return array
   *   The modified workflow JSON.
   */
  protected function applyFieldMappings(array $workflow_json, array $field_values): array {
    foreach ($field_values as $field_name => $value) {
      $mapping = $workflow_json['field_mappings'][$field_name] ?? NULL;
      if (!$mapping) {
        continue;
      }

      if (isset($mapping['node_id'], $mapping['parameter'])) {
        $workflow_json['nodes'][$mapping['node_id']][$mapping['parameter']] = $value;
      }
    }

    return $workflow_json;
  }

  /**
   * Processes the workflow execution result.
   *
   * @param object $workflow
   *   The workflow entity.
   * @param array $result
   *   The execution result.
   * @param array $data
   *   The original queue item data.
   *
   * @throws \RuntimeException
   *   When the result structure is invalid.
   */
  protected function processWorkflowResult($workflow, array $result, array $data): void {
    if (!$this->validateResult($result)) {
      throw new RuntimeException('Invalid workflow result structure');
    }

    // Save output files
    if (!empty($result['output'])) {
      $config = $this->configFactory->get('comfyui.settings');
      $output_path = $config->get('default_output_path');

      foreach ($result['output'] as $output_type => $output_data) {
        $saved_path = $this->saveOutput($output_data, $output_type, $output_path, $workflow->id());
        $this->notifyOutputAvailable($workflow, $saved_path);
      }
    }

    // Trigger any callbacks
    if (isset($data['callback']) && is_callable($data['callback'])) {
      call_user_func($data['callback'], $workflow, $result);
    }
  }

  /**
   * Validates the workflow result structure.
   *
   * @param array $result
   *   The result to validate.
   *
   * @return bool
   *   TRUE if the result is valid, FALSE otherwise.
   */
  protected function validateResult(array $result): bool {
    return isset($result['output']) && is_array($result['output']);
  }

  /**
   * Saves workflow output.
   *
   * @param mixed $output_data
   *   The output data.
   * @param string $output_type
   *   The type of output.
   * @param string $output_path
   *   The base output path.
   * @param int $workflow_id
   *   The workflow ID.
   *
   * @return string
   *   The saved file path.
   */
  protected function saveOutput($output_data, string $output_type, string $output_path, int $workflow_id): string {
    // Implement output saving logic based on type
    $file_path = $output_path . '/' . $workflow_id . '_' . time() . '_' . $output_type;
    
    // Log the save operation
    $this->loggerFactory->get('comfyui')
      ->info('Saved @type output for workflow @id at @path', [
        '@type' => $output_type,
        '@id' => $workflow_id,
        '@path' => $file_path,
      ]);

    return $file_path;
  }

  /**
   * Notifies that output is available.
   *
   * @param object $workflow
   *   The workflow entity.
   * @param string $file_path
   *   The path to the output file.
   */
  protected function notifyOutputAvailable($workflow, string $file_path): void {
    // Implement notification logic
    // This is a placeholder that still needs implementing
  }

}
