<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for fetching ComfyUI API options using object_info endpoint.
 */
class ComfyUIApiOptionsService {

  /**
   * The HTTP client.
   */
  protected $httpClient;

  /**
   * The config factory.
   */
  protected $configFactory;

  /**
   * The logger factory.
   */
  protected $loggerFactory;

  /**
   * The cache backend.
   */
  protected $cache;

  /**
   * Cache TTL in seconds (1 hour).
   */
  protected const CACHE_TTL = 3600;

  /**
   * Constructs a ComfyUIApiOptionsService object.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    CacheBackendInterface $cache
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->cache = $cache;
  }

  /**
   * Fetches object info for a specific node type.
   *
   * @param string $class_type
   *   The ComfyUI node class type.
   *
   * @return array|null
   *   The object info data or NULL if failed.
   */
  /**
   * Fetches object info for a specific node type.
   *
   * @param string $class_type
   *   The ComfyUI node class type.
   *
   * @return array|null
   *   The object info data or NULL if failed.
   */
  public function getObjectInfo($class_type) {
    $cache_key = "comfyui_object_info:{$class_type}";
    
    // Try cache first
    if ($cached = $this->cache->get($cache_key)) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('comfyui.settings');
      $api_endpoint = $config->get('api_endpoint');
      $timeout = $config->get('timeout') ?: 30;

      // Build request options with authentication
      $request_options = [
        'timeout' => $timeout,
        'headers' => ['Accept' => 'application/json'],
      ];
      
      // Add authentication if login module is enabled
      if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
        $token_service = \Drupal::service('comfyui_login.token_service');
        $request_options = $token_service->buildRequestOptions($request_options);
      }

      $response = $this->httpClient->get($api_endpoint . '/object_info/' . urlencode($class_type), $request_options);

      $data = json_decode($response->getBody(), TRUE);
      
      if (json_last_error() === JSON_ERROR_NONE && !empty($data)) {
        // Cache with tags for easier bulk clearing
        $this->cache->set(
          $cache_key, 
          $data, 
          time() + self::CACHE_TTL,
          ['comfyui_api_options', "comfyui_node_{$class_type}"]
        );
        
        return $data;
      }

    } catch (RequestException $e) {
      $this->loggerFactory->get('comfyui')
        ->warning('Failed to fetch object_info for @class_type: @error', [
          '@class_type' => $class_type,
          '@error' => $e->getMessage(),
        ]);
    }

    return NULL;
  }

  /**
   * Fetches object info for multiple node types.
   *
   * @param array $class_types
   *   Array of ComfyUI node class types.
   *
   * @return array
   *   Array keyed by class_type with object info data.
   */
  public function getMultipleObjectInfo(array $class_types) {
    $results = [];
    
    foreach ($class_types as $class_type) {
      $object_info = $this->getObjectInfo($class_type);
      if ($object_info) {
        $results[$class_type] = $object_info;
      }
    }
    
    return $results;
  }

  /**
   * Extracts available options for a specific input from object info.
   *
   * @param array $object_info
   *   The object info data for a node type.
   * @param string $input_name
   *   The input field name.
   *
   * @return array
   *   Array of available options for the input.
   */
  public function extractInputOptions($object_info, $input_name) {
    // The object_info response wraps the data with the node type as the key
    // We need to get the first (and only) value from the response
    $node_data = reset($object_info);
    
    if (!$node_data || !isset($node_data['input'])) {
      $this->loggerFactory->get('comfyui')
        ->debug('No input data found in object_info structure');
      return [];
    }

    $input_data = $node_data['input'];
    
    // Check required inputs first, then optional
    $input_def = $input_data['required'][$input_name] ?? 
                $input_data['optional'][$input_name] ?? 
                [];

    if (empty($input_def) || !is_array($input_def)) {
      $this->loggerFactory->get('comfyui')
        ->debug('Input @input not found. Available required: @req, optional: @opt', [
          '@input' => $input_name,
          '@req' => json_encode(array_keys($input_data['required'] ?? [])),
          '@opt' => json_encode(array_keys($input_data['optional'] ?? [])),
        ]);
      return [];
    }

    // The first element contains the options array (for select inputs)
    $options = $input_def[0] ?? [];
    
    // Only return options if it's actually an array of choices
    if (is_array($options) && !empty($options) && is_string($options[0] ?? '')) {
      $this->loggerFactory->get('comfyui')
        ->debug('Found @count options for @input: @options', [
          '@count' => count($options),
          '@input' => $input_name,
          '@options' => json_encode(array_slice($options, 0, 5)) . (count($options) > 5 ? '...' : ''),
        ]);
      
      return $options;
    }
    
    // For non-select inputs (INT, FLOAT, STRING), return empty array
    return [];
  }

  /**
   * Tests if the ComfyUI API is accessible.
   *
   * @return bool
   *   TRUE if API is accessible, FALSE otherwise.
   */
  public function testApiConnection() {
    try {
      $config = $this->configFactory->get('comfyui.settings');
      $api_endpoint = $config->get('api_endpoint');

      // Build request options with authentication
      $request_options = [
        'timeout' => 5,
        'connect_timeout' => 5,
      ];
      
      // Add authentication if login module is enabled
      if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
        $token_service = \Drupal::service('comfyui_login.token_service');
        $request_options = $token_service->buildRequestOptions($request_options);
      }

      $response = $this->httpClient->get($api_endpoint, $request_options);

      return $response->getStatusCode() === 200;
    } catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Clears cached object info data.
   */
  public function clearCache($class_type = NULL) {
    if ($class_type) {
      $cache_key = "comfyui_object_info:{$class_type}";
      $this->cache->delete($cache_key);
      
      $this->loggerFactory->get('comfyui')
        ->info('Cleared cache for @class_type', ['@class_type' => $class_type]);
    } else {
      // Clear using multiple methods for reliability
      if (method_exists($this->cache, 'invalidateTags')) {
        try {
          $this->cache->invalidateTags(['comfyui_api_options']);
        } catch (\Exception $e) {
          // Silent fail, try other methods
        }
      }
      
      try {
        $this->cache->deleteAll();
      } catch (\Exception $e) {
        // Silent fail
      }
      
      // Manual deletion as fallback
      $known_node_types = ['CheckpointLoaderSimple', 'KSampler', 'LoraLoader', 'VAELoader', 'ControlNetLoader'];
      foreach ($known_node_types as $node_type) {
        $this->cache->delete("comfyui_object_info:{$node_type}");
      }
      
      $this->loggerFactory->get('comfyui')
        ->info('Cleared all ComfyUI API cache entries');
    }
  }

}
