<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for managing ComfyUI node and group bypass functionality.
 * 
 * Handles:
 * - Parsing groups from workflow.json using spatial bounding box logic
 * - Validating group names and node IDs
 * - Building bypass directives based on Boolean field values
 * - Converting bypass mappings into node mode:4 settings
 */
class ComfyUIBypassMappingService {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a ComfyUIBypassMappingService object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory) {
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Parse groups from workflow.json and extract group names.
   *
   * @param string $workflow_json_content
   *   The raw workflow.json file content (UI format with groups).
   *
   * @return array
   *   Array of group names with their bounding boxes.
   *   Format: ['Group Name' => ['bounding' => [...], 'id' => 3, ...], ...]
   *
   * @throws \InvalidArgumentException
   *   If JSON is invalid or malformed.
   */
  public function parseGroupsFromWorkflow($workflow_json_content) {
    $workflow_data = json_decode($workflow_json_content, TRUE);

    if (json_last_error() !== JSON_ERROR_NONE) {
      throw new \InvalidArgumentException(
        'Invalid JSON format in workflow file: ' . json_last_error_msg()
      );
    }

    if (empty($workflow_data['groups'])) {
      return [];
    }

    $groups = [];
    foreach ($workflow_data['groups'] as $group) {
      $title = $group['title'] ?? 'Unnamed Group';
      $groups[$title] = [
        'bounding' => $group['bounding'] ?? [],
        'id' => $group['id'] ?? NULL,
        'color' => $group['color'] ?? '#3f789e',
        'font_size' => $group['font_size'] ?? 24,
      ];
    }

    $this->loggerFactory->get('comfyui')
      ->debug('Parsed @count groups from workflow', ['@count' => count($groups)]);

    return $groups;
  }

  /**
   * Get all nodes that belong to a specific group by spatial bounds checking.
   *
   * @param array $workflow_data
   *   The parsed workflow.json data (UI format with nodes array).
   * @param string $group_name
   *   The name of the group to extract nodes from.
   *
   * @return array
   *   Array of node IDs that fall within the group's bounding box.
   *   Returns empty array if group not found.
   *
   * @throws \InvalidArgumentException
   *   If workflow_data is invalid or missing nodes.
   */
  public function getNodesInGroup(array $workflow_data, $group_name) {
    if (empty($workflow_data['groups'])) {
      return [];
    }

    // Find the group by title
    $target_group = NULL;
    foreach ($workflow_data['groups'] as $group) {
      if ($group['title'] === $group_name) {
        $target_group = $group;
        break;
      }
    }

    if (!$target_group) {
      $this->loggerFactory->get('comfyui')
        ->warning('Group "@name" not found in workflow', ['@name' => $group_name]);
      return [];
    }

    // Extract bounding box
    $bounding = $target_group['bounding'] ?? [];
    if (empty($bounding) || count($bounding) < 4) {
      $this->loggerFactory->get('comfyui')
        ->warning('Invalid bounding box for group "@name"', ['@name' => $group_name]);
      return [];
    }

    // Convert ComfyUI format [x1, y1, width, height] to min/max coordinates
    list($x1, $y1, $width, $height) = $bounding;
    $x2 = $x1 + $width;
    $y2 = $y1 + $height;

    $nodes_in_group = [];

    // Check each node's position against the bounding box
    foreach ($workflow_data['nodes'] as $node) {
      $node_id = $node['id'] ?? NULL;
      $pos = $node['pos'] ?? NULL;

      if (!$node_id || !$pos || count($pos) < 2) {
        continue;
      }

      list($node_x, $node_y) = $pos;

      // Point-in-box check using inclusive bounds
      if ($node_x >= $x1 && $node_x <= $x2 && $node_y >= $y1 && $node_y <= $y2) {
        $nodes_in_group[] = (string) $node_id;
      }
    }

    $this->loggerFactory->get('comfyui')
      ->info('Found @count nodes in group "@name"', [
        '@count' => count($nodes_in_group),
        '@name' => $group_name,
      ]);

    return $nodes_in_group;
  }

  /**
   * Validate that a group exists in the workflow.
   *
   * @param array $workflow_data
   *   The parsed workflow.json data.
   * @param string $group_name
   *   The group name to validate.
   *
   * @return bool
   *   TRUE if group exists, FALSE otherwise.
   */
  public function validateGroupExists(array $workflow_data, $group_name) {
    if (empty($workflow_data['groups'])) {
      return FALSE;
    }

    foreach ($workflow_data['groups'] as $group) {
      if ($group['title'] === $group_name) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Validate that nodes exist in the workflow API data.
   *
   * @param array $workflow_api_data
   *   The parsed workflow_api.json data (API execution format).
   * @param array $node_ids
   *   Array of node IDs to validate.
   *
   * @return array
   *   Array of validation results with node IDs as keys and TRUE/FALSE as values.
   */
  public function validateNodeIdsExist(array $workflow_api_data, array $node_ids) {
    $results = [];

    foreach ($node_ids as $node_id) {
      $node_id_str = (string) $node_id;
      $exists = isset($workflow_api_data[$node_id_str]);
      $results[$node_id_str] = $exists;

      if (!$exists) {
        $this->loggerFactory->get('comfyui')
          ->warning('Node ID "@id" not found in workflow API', ['@id' => $node_id_str]);
      }
    }

    return $results;
  }

  /**
   * Build bypass directives from Boolean field values and bypass mappings.
   */
  public function buildBypassDirectives($workflow, array $input_values, array $workflow_data) {
    $bypass_mappings = $workflow->getBypassMappings();
    
    $nodes_to_bypass = [];

    if (empty($bypass_mappings)) {
      return $nodes_to_bypass;
    }

    if (!empty($bypass_mappings['group_bypasses'])) {
      $nodes_to_bypass = array_merge(
        $nodes_to_bypass,
        $this->processGroupBypasses($bypass_mappings['group_bypasses'], $input_values, $workflow_data)
      );
    }

    if (!empty($bypass_mappings['node_bypasses'])) {
      $nodes_to_bypass = array_merge(
        $nodes_to_bypass,
        $this->processNodeBypasses($bypass_mappings['node_bypasses'], $input_values)
      );
    }

    $nodes_to_bypass = array_unique($nodes_to_bypass);

    return $nodes_to_bypass;
  }

  /**
   * Process group bypass mappings to extract nodes to bypass.
   */
  protected function processGroupBypasses(array $group_bypasses, array $input_values, array $workflow_data) {
    $nodes_to_bypass = [];

    foreach ($group_bypasses as $field_name => $bypass_config) {
      $field_value = $input_values[$field_name] ?? TRUE;
      
      // Handle string "false" from JavaScript
      if ($field_value === "false" || $field_value === "0") {
        $field_value = false;
      }
      
      $is_enabled = (bool) $field_value;

      if (!$is_enabled) {
        $node_ids = $bypass_config['node_ids'] ?? [];
        $nodes_to_bypass = array_merge($nodes_to_bypass, $node_ids);
      }
    }

    return $nodes_to_bypass;
  }

  /**
   * Process node bypass mappings to extract nodes to bypass.
   */
  protected function processNodeBypasses(array $node_bypasses, array $input_values) {
    $nodes_to_bypass = [];

    foreach ($node_bypasses as $field_name => $node_ids_string) {
      $field_value = $input_values[$field_name] ?? TRUE;
      $is_enabled = (bool) $field_value;

      if (!$is_enabled) {
        // Handle both string and array formats
        if (is_array($node_ids_string)) {
          $node_ids = array_map('trim', array_map('strval', $node_ids_string));
        } else {
          $node_ids = array_filter(array_map('trim', explode(',', $node_ids_string)));
        }
        $nodes_to_bypass = array_merge($nodes_to_bypass, $node_ids);
      }
    }

    return $nodes_to_bypass;
  }

  /**
   * Apply bypass mode (mode: 4) to specified nodes in workflow API data.
   * 
   * Note: This method is deprecated - rewiring is now used instead.
   * Kept for backward compatibility only.
   */
  public function applyNodeBypasses(array &$workflow_api_data, array $nodes_to_bypass) {
    $bypassed_count = 0;

    foreach ($nodes_to_bypass as $node_id) {
      $node_id_str = (string) $node_id;

      if (!isset($workflow_api_data[$node_id_str])) {
        continue;
      }

      $workflow_api_data[$node_id_str]['mode'] = 4;
      $bypassed_count++;
    }

    return $bypassed_count;
  }

  /**
   * Get available Boolean fields from an entity.
   *
   * @param object $entity
   *   The entity to check for Boolean fields.
   *
   * @return array
   *   Array of Boolean field names with their labels.
   *   Format: ['field_enable_upscaling' => 'Enable Upscaling', ...]
   */
  public function getAvailableBooleanFields($entity) {
    $field_definitions = \Drupal::service('entity_field.manager')
      ->getFieldDefinitions($entity->getEntityTypeId(), $entity->bundle());

    $boolean_fields = [];
    $skip_fields = [
      'id', 'uuid', 'title', 'description', 'workflow_json', 'workflow_api',
      'field_mappings', 'bypass_mappings', 'discovered_fields', 'status',
      'created', 'changed', 'node_mappings', 'model_mappings',
      'discovered_nodes', 'discovered_models',
    ];

    foreach ($field_definitions as $field_name => $field_definition) {
      if (in_array($field_name, $skip_fields)) {
        continue;
      }

      // Only include configurable boolean fields
      if ($field_definition instanceof \Drupal\Core\Field\FieldConfigInterface &&
          $field_definition->getType() === 'boolean') {
        $boolean_fields[$field_name] = (string) $field_definition->getLabel();
      }
    }

    return $boolean_fields;
  }

}
