<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;

/**
 * Helper service for managing media reference fields on workflows.
 */
class ComfyUIMediaFieldHelper {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * Constructs a ComfyUIMediaFieldHelper object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityFieldManagerInterface $entity_field_manager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
  }

  /**
   * Create a media reference field for input images.
   *
   * @param string $field_name
   *   The machine name for the field.
   * @param string $field_label
   *   The human-readable label for the field.
   * @param int $cardinality
   *   Number of values (1 for single, -1 for unlimited).
   *
   * @return \Drupal\field\Entity\FieldConfig|null
   *   The created field config, or NULL on failure.
   */
  public function createMediaReferenceField($field_name, $field_label, $cardinality = 1) {
    try {
      // Check if storage exists
      $storage_id = 'comfyui_workflow.' . $field_name;
      $field_storage = FieldStorageConfig::load($storage_id);
      
      if (!$field_storage) {
        // Create field storage
        $field_storage = FieldStorageConfig::create([
          'field_name' => $field_name,
          'entity_type' => 'comfyui_workflow',
          'type' => 'entity_reference',
          'cardinality' => $cardinality,
          'settings' => [
            'target_type' => 'media',
          ],
        ]);
        $field_storage->save();
      }

      // Check if field instance exists
      $field = FieldConfig::load('comfyui_workflow.comfyui_workflow.' . $field_name);
      
      if (!$field) {
        // Create field instance
        $field = FieldConfig::create([
          'field_storage' => $field_storage,
          'bundle' => 'comfyui_workflow',
          'label' => $field_label,
          'description' => t('Select media entity to use as input for this workflow.'),
          'required' => FALSE,
          'settings' => [
            'handler' => 'default:media',
            'handler_settings' => [
              'target_bundles' => [
                'comfyui_input' => 'comfyui_input',
                'comfyui_generated' => 'comfyui_generated',
              ],
              'auto_create' => FALSE,
            ],
          ],
        ]);
        $field->save();

        // Set form display to use media library widget
        $form_display = $this->entityTypeManager
          ->getStorage('entity_form_display')
          ->load('comfyui_workflow.comfyui_workflow.default');
        
        if ($form_display) {
          $form_display->setComponent($field_name, [
            'type' => 'media_library_widget',
            'weight' => 10,
            'settings' => [
              'media_types' => ['comfyui_input', 'comfyui_generated'],
            ],
          ]);
          $form_display->save();
        }

        // Set view display
        $view_display = $this->entityTypeManager
          ->getStorage('entity_view_display')
          ->load('comfyui_workflow.comfyui_workflow.default');
        
        if ($view_display) {
          $view_display->setComponent($field_name, [
            'type' => 'entity_reference_label',
            'weight' => 10,
            'label' => 'above',
          ]);
          $view_display->save();
        }
      }

      return $field;

    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Failed to create media reference field: @error', [
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Check if a field is a media reference field.
   *
   * @param string $field_name
   *   The field name.
   *
   * @return bool
   *   TRUE if it's a media reference field.
   */
  public function isMediaReferenceField($field_name) {
    $field_definitions = $this->entityFieldManager
      ->getFieldDefinitions('comfyui_workflow', 'comfyui_workflow');
    
    if (!isset($field_definitions[$field_name])) {
      return FALSE;
    }

    $field_definition = $field_definitions[$field_name];
    
    if ($field_definition->getType() !== 'entity_reference') {
      return FALSE;
    }

    $target_type = $field_definition->getSetting('target_type');
    return $target_type === 'media';
  }

}
