<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\PrivateKey;
use Drupal\Core\Site\Settings;
use Drupal\media\MediaInterface;
use Drupal\file\FileInterface;

/**
 * Service for generating temporary tokenized URLs for private media files.
 */
class ComfyUIMediaTokenService {

  /**
   * The private key service.
   */
  protected $privateKey;

  /**
   * Default token expiration time in seconds (1 hour).
   */
  const DEFAULT_EXPIRATION = 3600;

  /**
   * Constructs a ComfyUIMediaTokenService object.
   */
  public function __construct(PrivateKey $private_key) {
    $this->privateKey = $private_key;
  }

  /**
   * Generate a temporary tokenized URL for a media entity.
   */
  public function generateTemporaryUrl(MediaInterface $media, $expiration = 3600) {
    try {
      if (!$media->hasField('field_media_image')) {
        return NULL;
      }
      
      $file = $media->get('field_media_image')->entity;
      if (!$file) {
        return NULL;
      }
      
      // Store file URI in default cache with unique token ID
      $token_id = bin2hex(random_bytes(16));
      $file_uri = $file->getFileUri();
      
      // Use default cache (not comfyui_tokens)
      \Drupal::cache()->set('comfyui_token_' . $token_id, $file_uri, time() + $expiration);
      
      // Return proxy URL with token_id
      $url = \Drupal\Core\Url::fromRoute('comfyui.image_proxy', [
        'filename' => basename($file_uri),
      ], [
        'query' => [
          'token_id' => $token_id,
        ],
        'absolute' => TRUE,
      ])->toString();
      
      return $url;
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->warning(
        'Failed to generate URL for media @id: @error',
        ['@id' => $media->id(), '@error' => $e->getMessage()]
      );
      return NULL;
    }
  }

  /**
   * Validate a token for a given URI and expiration time.
   */
  public function validateToken($uri, $token, $expires) {
    if (time() > $expires) {
      return FALSE;
    }

    $expected_token = $this->generateToken($uri, $expires);
    return hash_equals($expected_token, $token);
  }

  /**
   * Generate a token for a given URI and expiration time.
   */
  protected function generateToken($uri, $expires) {
    $secret = $this->privateKey->get() . Settings::getHashSalt();
    // Use SAME format as controller: uri|expires
    $data = $uri . '|' . $expires;
    
    return hash_hmac('sha256', $data, $secret);
  }

  /**
   * Get the file entity from a media entity.
   */
  protected function getFileFromMedia(MediaInterface $media) {
    $source = $media->getSource();
    $source_field = $source->getConfiguration()['source_field'];
    
    if (!$media->hasField($source_field)) {
      return NULL;
    }

    $file_field = $media->get($source_field);
    
    if ($file_field->isEmpty()) {
      return NULL;
    }

    return $file_field->entity;
  }

}
