<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for mapping ComfyUI models to download sources.
 */
class ComfyUIModelMappingService {

  /**
   * The HTTP client.
   */
  protected $httpClient;

  /**
   * The config factory.
   */
  protected $configFactory;

  /**
   * The logger factory.
   */
  protected $loggerFactory;

  /**
   * The cache backend.
   */
  protected $cache;

  /**
   * Cache TTL in seconds (1 hour).
   */
  protected const CACHE_TTL = 3600;

  /**
   * Constructs a ComfyUIModelMappingService object.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    CacheBackendInterface $cache
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->cache = $cache;
  }

  /**
   * Fetches the model-list.json from ComfyUI Manager.
   */
  public function fetchModelList() {
    $cache_key = 'comfyui_model_list';
    
    if ($cached = $this->cache->get($cache_key)) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('comfyui.settings');
      $url = $config->get('manager_model_list_url') ?: 
        'https://raw.githubusercontent.com/Comfy-Org/ComfyUI-Manager/main/model-list.json';

      $response = $this->httpClient->get($url, ['timeout' => 30]);
      $data = json_decode($response->getBody(), TRUE);

      if (json_last_error() === JSON_ERROR_NONE && !empty($data)) {
        $this->cache->set($cache_key, $data, time() + self::CACHE_TTL, ['comfyui_model_mapping']);
        return $data;
      }
    } catch (RequestException $e) {
      $this->loggerFactory->get('comfyui')
        ->error('Failed to fetch model-list.json: @error', [
          '@error' => $e->getMessage(),
        ]);
    }

    return ['models' => []];
  }

  /**
   * Gets available model types from model-list.json.
   */
  public function getAvailableModelTypes() {
    $model_list = $this->fetchModelList();
    $types = [];

    foreach ($model_list['models'] ?? [] as $model) {
      if (!empty($model['type'])) {
        $types[$model['type']] = $model['type'];
      }
    }

    return array_values($types);
  }

  /**
   * Extracts model references from workflow data by scanning for model filenames.
   */
  public function extractModelReferences(array $api_data) {
    $model_references = [];

    foreach ($api_data as $node_id => $node) {
      $class_type = $node['class_type'] ?? '';
      $inputs = $node['inputs'] ?? [];
      
      // Scan ALL inputs for values that look like model files
      foreach ($inputs as $input_key => $input_value) {
        // Skip array values (node connections)
        if (!is_string($input_value) || empty($input_value)) {
          continue;
        }
        
        // Check if this looks like a model file
        if ($this->looksLikeModelFile($input_value)) {
          $model_references[$input_value] = [
            'filename' => $input_value,
            'loader_type' => $class_type,
            'node_id' => $node_id,
            'input_key' => $input_key,
          ];
        }
      }
    }

    $this->loggerFactory->get('comfyui')->info(
      'Extracted @count model references from workflow by scanning for file extensions',
      ['@count' => count($model_references)]
    );

    return $model_references;
  }

  /**
   * Finds model information in model-list.json by filename.
   */
  public function findModelInfo($filename) {
    $model_list = $this->fetchModelList();
    
    // Extract basename from path (e.g., "SDXL/union/model.safetensors" -> "model.safetensors")
    $basename = basename($filename);
    
    foreach ($model_list['models'] ?? [] as $model) {
      if (isset($model['filename'])) {
        // Compare both full path and basename
        if ($model['filename'] === $filename || $model['filename'] === $basename) {
          return $model;
        }
      }
    }

    return NULL;
  }

  /**
   * Discovers models with download information.
   */
  public function discoverModelsWithDownloads(array $api_data) {
    $model_references = $this->extractModelReferences($api_data);
    $discovered = [];

    foreach ($model_references as $filename => $ref_info) {
      $model_info = $this->findModelInfo($filename);
      
      if ($model_info) {
        // Found in database
        $save_path = $model_info['save_path'] ?? '';
        
        // Normalize save_path
        if (empty($save_path) || $save_path === 'default') {
          // Incomplete data - leave blank for manual entry
          $save_path = '';
        } else {
          // Add models/ prefix if missing
          if (strpos($save_path, 'models/') !== 0) {
            $save_path = 'models/' . ltrim($save_path, '/');
          }
          
          // Ensure trailing slash
          if (substr($save_path, -1) !== '/') {
            $save_path .= '/';
          }
        }
        
        $discovered[$filename] = [
          'filename' => $filename,
          'type' => $model_info['type'] ?? 'unknown',
          'base' => $model_info['base'] ?? '',
          'save_path' => $save_path,
          'urls' => is_array($model_info['url']) ? $model_info['url'] : [$model_info['url']],
          'reference' => $model_info['reference'] ?? '',
          'source' => 'comfyui_manager',
          'loader_type' => $ref_info['loader_type'],
        ];
      } else {
        // Not found - infer from model-list.json patterns
        $inferred = $this->inferTypeAndPath($ref_info['loader_type'], $filename);
        
        $discovered[$filename] = [
          'filename' => $filename,
          'type' => $inferred['type'],
          'base' => '',
          'save_path' => $inferred['save_path'],
          'urls' => [],
          'reference' => '',
          'source' => 'manual',
          'loader_type' => $ref_info['loader_type'],
        ];
      }
    }

    return $discovered;
  }

  /**
   * Infer type and save path by analyzing model-list.json patterns.
   * 
   * Looks at other models in the database to learn patterns instead of hardcoding.
   */
  protected function inferTypeAndPath($loader_type, $filename) {
    $model_list = $this->fetchModelList();
    
    // Analyze model-list.json to find common patterns
    $type_analysis = [];
    
    foreach ($model_list['models'] ?? [] as $model) {
      $type = $model['type'] ?? '';
      $save_path = $model['save_path'] ?? '';
      
      if (empty($type)) {
        continue;
      }
      
      // Track what save_paths are used for each type
      if (!isset($type_analysis[$type])) {
        $type_analysis[$type] = [
          'save_paths' => [],
          'count' => 0,
        ];
      }
      
      $type_analysis[$type]['count']++;
      if (!empty($save_path) && $save_path !== 'default') {
        $type_analysis[$type]['save_paths'][] = $save_path;
      }
    }
    
    // Try to match loader name to a type in the database
    // e.g., "UltralyticsDetectorProvider" -> look for type "Ultralytics"
    $inferred_type = null;
    $inferred_path = '';
    
    foreach ($type_analysis as $type => $data) {
      // Check if loader name contains this type
      if (stripos($loader_type, $type) !== FALSE) {
        $inferred_type = $type;
        
        // Get most common save_path for this type
        if (!empty($data['save_paths'])) {
          $path_counts = array_count_values($data['save_paths']);
          arsort($path_counts);
          $most_common_path = key($path_counts);
          
          // Normalize it
          if (strpos($most_common_path, 'models/') !== 0) {
            $most_common_path = 'models/' . ltrim($most_common_path, '/');
          }
          
          // Strip subdirectories - only keep base type folder
          // e.g., "models/ultralytics/bbox/" -> "models/ultralytics/"
          // e.g., "models/ultralytics/segm/" -> "models/ultralytics/"
          // The filename already contains subdirectory structure
          $path_parts = explode('/', trim($most_common_path, '/'));
          if (count($path_parts) > 2) {
            // Keep only models/type_name/
            $most_common_path = $path_parts[0] . '/' . $path_parts[1] . '/';
          } else {
            // Ensure trailing slash
            if (substr($most_common_path, -1) !== '/') {
              $most_common_path .= '/';
            }
          }
          
          $inferred_path = $most_common_path;
        }
        
        break;
      }
    }
    
    // If no match found, try generic extraction from loader name
    if (!$inferred_type && preg_match('/^(.+?)(?:Loader|Provider|Detector)/', $loader_type, $matches)) {
      $inferred_type = $matches[1];
    }
    
    return [
      'type' => $inferred_type ?: 'unknown',
      'save_path' => $inferred_path,
    ];
  }

  /**
   * Clears model mapping cache.
   */
  public function clearCache() {
    $this->cache->delete('comfyui_model_list');
    
    $this->loggerFactory->get('comfyui')->info('Cleared model mapping cache');
  }

  /**
   * Checks if a string looks like a model filename.
   */
  protected function looksLikeModelFile($filename) {
    // Must be a reasonable length
    if (strlen($filename) < 3) {
      return FALSE;
    }
    
    // Model files typically have file extensions
    $valid_extensions = [
      'safetensors',
      'ckpt',
      'pt',
      'pth',
      'bin',
      'onnx',
      'pb',
      'trt',
    ];
    
    $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
    
    // Must have a valid extension
    return in_array($extension, $valid_extensions);
  }

}
