<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for mapping ComfyUI node types to repositories.
 */
class ComfyUINodeMappingService {

  /**
   * The HTTP client.
   */
  protected $httpClient;

  /**
   * The config factory.
   */
  protected $configFactory;

  /**
   * The logger factory.
   */
  protected $loggerFactory;

  /**
   * The cache backend.
   */
  protected $cache;

  /**
   * Cache TTL in seconds (1 hour).
   */
  protected const CACHE_TTL = 3600;

  /**
   * Constructs a ComfyUINodeMappingService object.
   */
  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    CacheBackendInterface $cache
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->cache = $cache;
  }

  /**
   * Fetches core/built-in node types from ComfyUI API.
   */
  public function fetchCoreNodeTypes() {
    $cache_key = 'comfyui_core_node_types';
    
    if ($cached = $this->cache->get($cache_key)) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('comfyui.settings');
      $api_endpoint = $config->get('api_endpoint');
      
      if (!$api_endpoint) {
        throw new \Exception('ComfyUI API endpoint not configured');
      }

      // Build request options with authentication if needed
      $request_options = [
        'timeout' => 10,
        'headers' => ['Accept' => 'application/json'],
      ];
      
      if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
        $token_service = \Drupal::service('comfyui_login.token_service');
        $request_options = $token_service->buildRequestOptions($request_options);
      }

      // Fetch /object_info to get all available nodes
      $response = $this->httpClient->get($api_endpoint . '/object_info', $request_options);
      $data = json_decode($response->getBody(), TRUE);

      if (json_last_error() === JSON_ERROR_NONE && !empty($data)) {
        $core_nodes = array_keys($data);
        
        $this->cache->set(
          $cache_key, 
          $core_nodes, 
          time() + self::CACHE_TTL,
          ['comfyui_node_mapping']
        );
        
        $this->loggerFactory->get('comfyui')->info(
          'Fetched @count core node types from ComfyUI API',
          ['@count' => count($core_nodes)]
        );
        
        return $core_nodes;
      }
    } catch (RequestException $e) {
      $this->loggerFactory->get('comfyui')
        ->warning('Failed to fetch core nodes from API: @error. Using fallback list.', [
          '@error' => $e->getMessage(),
        ]);
    }

    // Fallback to minimal essential nodes if API is unavailable
    return $this->getFallbackCoreNodes();
  }

  /**
   * Provides fallback core nodes when API is unavailable.
   */
  protected function getFallbackCoreNodes() {
    return [
      'KSampler',
      'KSamplerAdvanced',
      'CheckpointLoaderSimple',
      'CheckpointLoader',
      'VAEDecode',
      'VAEEncode',
      'VAELoader',
      'CLIPTextEncode',
      'CLIPLoader',
      'CLIPSetLastLayer',
      'ControlNetLoader',
      'ControlNetApply',
      'PreviewImage',
      'SaveImage',
      'LoadImage',
      'InvertMask',
      'SetLatentNoiseMask',
      'LatentUpscale',
      'EmptyLatentImage',
    ];
  }

  /**
   * Fetches the extension-node-map.json from ComfyUI Manager.
   */
  public function fetchNodeMap() {
    $cache_key = 'comfyui_extension_node_map';
    
    if ($cached = $this->cache->get($cache_key)) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('comfyui.settings');
      $url = $config->get('manager_node_map_url') ?: 
        'https://raw.githubusercontent.com/Comfy-Org/ComfyUI-Manager/main/extension-node-map.json';

      $response = $this->httpClient->get($url, ['timeout' => 30]);
      $data = json_decode($response->getBody(), TRUE);

      if (json_last_error() === JSON_ERROR_NONE && !empty($data)) {
        $this->cache->set($cache_key, $data, time() + self::CACHE_TTL, ['comfyui_node_mapping']);
        return $data;
      }
    } catch (RequestException $e) {
      $this->loggerFactory->get('comfyui')
        ->error('Failed to fetch extension-node-map.json: @error', [
          '@error' => $e->getMessage(),
        ]);
    }

    return [];
  }

  /**
   * Fetches the custom-node-list.json from ComfyUI Manager.
   */
  public function fetchCustomNodeList() {
    $cache_key = 'comfyui_custom_node_list';
    
    if ($cached = $this->cache->get($cache_key)) {
      return $cached->data;
    }

    try {
      $config = $this->configFactory->get('comfyui.settings');
      $url = $config->get('manager_custom_node_list_url') ?: 
        'https://raw.githubusercontent.com/Comfy-Org/ComfyUI-Manager/main/custom-node-list.json';

      $response = $this->httpClient->get($url, ['timeout' => 30]);
      $data = json_decode($response->getBody(), TRUE);

      if (json_last_error() === JSON_ERROR_NONE && !empty($data)) {
        $this->cache->set($cache_key, $data, time() + self::CACHE_TTL, ['comfyui_node_mapping']);
        return $data;
      }
    } catch (RequestException $e) {
      $this->loggerFactory->get('comfyui')
        ->error('Failed to fetch custom-node-list.json: @error', [
          '@error' => $e->getMessage(),
        ]);
    }

    return [];
  }

  /**
   * Identifies custom nodes from workflow data.
   */
  public function identifyCustomNodes(array $workflow_data, array $api_data) {
    $all_node_types = [];
    
    // Extract from API data (class_type)
    foreach ($api_data as $node_id => $node) {
      if (isset($node['class_type'])) {
        $all_node_types[$node['class_type']] = $node['class_type'];
      }
    }

    // Also check workflow data for additional info
    if (isset($workflow_data['nodes'])) {
      foreach ($workflow_data['nodes'] as $node) {
        if (isset($node['type'])) {
          $all_node_types[$node['type']] = $node['type'];
        }
      }
    }

    // Get core nodes from API
    $core_nodes = $this->fetchCoreNodeTypes();

    // Filter out core nodes - remaining are custom
    $custom_nodes = array_diff(array_keys($all_node_types), $core_nodes);

    $this->loggerFactory->get('comfyui')->info(
      'Identified @custom custom nodes out of @total total node types',
      [
        '@custom' => count($custom_nodes),
        '@total' => count($all_node_types),
      ]
    );

    return array_values($custom_nodes);
  }

  /**
   * Finds repository recommendations for a node type.
   */
  public function findRepositoriesForNode($node_type) {
    $node_map = $this->fetchNodeMap();
    $repositories = [];

    foreach ($node_map as $repo_url => $data) {
      if (isset($data[0]) && is_array($data[0])) {
        $node_types = $data[0];
        if (in_array($node_type, $node_types)) {
          $title = $data[1]['title_aux'] ?? basename($repo_url);
          $repositories[] = [
            'url' => $repo_url,
            'title' => $title,
          ];
        }
      }
    }

    return $repositories;
  }

  /**
   * Discovers all custom nodes and their recommendations.
   */
  public function discoverNodesWithRecommendations(array $workflow_data, array $api_data) {
    $custom_node_types = $this->identifyCustomNodes($workflow_data, $api_data);
    $core_nodes = $this->fetchCoreNodeTypes();
    $discovered = [];

    foreach ($custom_node_types as $node_type) {
      $repositories = $this->findRepositoriesForNode($node_type);
      
      // Check if this node type also exists in core (conflict scenario)
      $is_core_conflict = in_array($node_type, $core_nodes);
      
      $discovered[$node_type] = [
        'node_type' => $node_type,
        'is_custom' => TRUE,
        'is_core_conflict' => $is_core_conflict,
        'repositories' => $repositories,
        'repository_count' => count($repositories),
        'recommended_action' => $this->determineRecommendedAction($repositories, $is_core_conflict),
      ];
    }

    $this->loggerFactory->get('comfyui')->info(
      'Discovered @count custom node types', 
      ['@count' => count($discovered)]
    );

    return $discovered;
  }

  /**
   * Determines recommended action for a node.
   */
  protected function determineRecommendedAction($repositories, $is_core_conflict) {
    if (empty($repositories)) {
      return 'manual'; // No repos found, needs manual URL
    }
    
    if ($is_core_conflict) {
      return 'conflict'; // Exists in both core and custom nodes
    }
    
    if (count($repositories) === 1) {
      return 'single'; // Single repo, auto-recommend
    }
    
    return 'multiple'; // Multiple repos, user must choose
  }

  /**
   * Clears node mapping cache.
   */
  public function clearCache() {
    $this->cache->delete('comfyui_core_node_types');
    $this->cache->delete('comfyui_extension_node_map');
    $this->cache->delete('comfyui_custom_node_list');
    
    $this->loggerFactory->get('comfyui')->info('Cleared node mapping cache');
  }

}
