<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Service for detecting and handling different ComfyUI output types.
 */
class ComfyUIOutputTypeManager {

  /**
   * Detect output type from ComfyUI response data.
   */
  public function detectOutputType($output_data, $field_definition) {
    // Check if it's file-based output (has filename)
    if (is_array($output_data) && isset($output_data['filename'])) {
      $extension = strtolower(pathinfo($output_data['filename'], PATHINFO_EXTENSION));
      
      if (in_array($extension, ['png', 'jpg', 'jpeg', 'gif', 'webp'])) {
        return 'image_file';
      }
      
      if (in_array($extension, ['mp4', 'mov', 'avi', 'mkv', 'webm'])) {
        return 'video_file';
      }
      
      if (in_array($extension, ['wav', 'mp3', 'ogg', 'flac'])) {
        return 'audio_file';
      }
      
      return 'generic_file';
    }
    
    // Check if it's data-based output
    if (is_string($output_data) || is_numeric($output_data)) {
      return 'data_value';
    }
    
    if (is_array($output_data) && !isset($output_data['filename'])) {
      return 'structured_data';
    }
    
    return 'unknown';
  }

  /**
   * Get appropriate handler for output type.
   */
  public function getOutputHandler($output_type) {
    // Simple handler mapping for now
    $handlers = [
      'image_file' => new \Drupal\comfyui\Plugin\ComfyUIOutputHandler\ImageFileHandler([], '', ''),
      'video_file' => new \Drupal\comfyui\Plugin\ComfyUIOutputHandler\VideoFileHandler([], '', ''),
      'data_value' => new \Drupal\comfyui\Plugin\ComfyUIOutputHandler\DataValueHandler([], '', ''),
    ];
    
    return $handlers[$output_type] ?? NULL;
  }

}
