<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use GuzzleHttp\Exception\ClientException;

/**
 * Service for executing ComfyUI workflows.
 */
class ComfyUIWorkflowExecutionService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The workflow parser.
   *
   * @var \Drupal\comfyui\Service\ComfyUIWorkflowParser
   */
  protected $workflowParser;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * Default headers for API requests.
   *
   * @var array
   */
  protected $defaultHeaders = [
    'Accept' => 'application/json',
    'Content-Type' => 'application/json',
  ];

  /**
   * Constructs a new WorkflowExecutionService.
   */
  /**
   * The media token service.
   *
   * @var \Drupal\comfyui\Service\ComfyUIMediaTokenService
   */
  protected $mediaTokenService;

  /**
   * The bypass mapping service.
   *
   * @var \Drupal\comfyui\Service\ComfyUIBypassMappingService
   */
  protected $bypassMappingService;

  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
    ComfyUIWorkflowParser $workflow_parser,
    FileSystemInterface $file_system,
    ComfyUIMediaTokenService $media_token_service = NULL,
    $bypass_mapping_service = NULL
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->workflowParser = $workflow_parser;
    $this->fileSystem = $file_system;
    $this->mediaTokenService = $media_token_service ?: \Drupal::service('comfyui.media_token');
    $this->bypassMappingService = $bypass_mapping_service ?: \Drupal::service('comfyui.bypass_mapping');
  }

  /**
   * Build request options with authentication if login module is enabled.
   *
   * @param array $additional_options
   *   Additional options to merge with base options.
   *
   * @return array
   *   Complete request options array.
   */
  protected function buildRequestOptions(array $additional_options = []) {
    $options = array_merge([
      'headers' => $this->defaultHeaders,
    ], $additional_options);
    
    // Check if comfyui_login module is enabled and add authentication
    if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
      $token_service = \Drupal::service('comfyui_login.token_service');
      $options = $token_service->buildRequestOptions($options);
    }
    
    return $options;
  }

  /**
   * Executes a workflow with error handling.
   *
   * @param int $workflow_id
   *   The workflow entity ID.
   * @param array $input_values
   *   Optional array of input values to override workflow defaults.
   *
   * @return array
   *   The execution result array.
   */
  public function executeWorkflow($workflow_id, array $input_values = []) {
    try {
      // Load and parse the workflow
      $workflow_data = $this->loadWorkflowData($workflow_id);
      
      // Apply any custom input values
      if (!empty($input_values)) {
        $workflow_data = $this->applyInputValues($workflow_data, $input_values);
      }

      // Execute the workflow
      return $this->attemptExecution($workflow_data);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('comfyui')
        ->error('Workflow execution failed: @error', [
          '@error' => $e->getMessage(),
        ]);

      return [
        'success' => FALSE,
        'message' => 'Workflow execution failed',
        'error' => $e->getMessage(),
      ];
    }
  }

  /**
   * Loads and validates workflow data.
   *
   * @param int $workflow_id
   *   The workflow entity ID.
   *
   * @return array
   *   The workflow data.
   *
   * @throws \InvalidArgumentException
   *   If the workflow or its files cannot be found.
   */
  protected function loadWorkflowData($workflow_id) {
    $workflow = $this->entityTypeManager
      ->getStorage('comfyui_workflow')
      ->load($workflow_id);

    if (!$workflow) {
      throw new \InvalidArgumentException("Workflow ID {$workflow_id} not found");
    }

    // USE workflow_api.json - it's already in the correct format for /prompt endpoint!
    $workflow_file = $workflow->get('workflow_api')->entity;
    if (!$workflow_file) {
      throw new \InvalidArgumentException("Workflow API JSON file not found for workflow ID {$workflow_id}");
    }

    $file_uri = $workflow_file->getFileUri();
    $real_path = $this->fileSystem->realpath($file_uri);

    return $this->workflowParser->parseWorkflowJson($real_path);
  }

  /**
   * Convert full workflow to prompt-compatible format.
   * Preserves spatial data (pos, size) needed for bypass (mode: 4) to work in ComfyUI.
   */
  protected function convertWorkflowToPromptFormat($workflow) {
    $prompt = [];
    
    // Check if already in dict format (node IDs as keys)
    if (isset($workflow['nodes']) && is_array($workflow['nodes'])) {
      // workflow.json format - nodes is an ARRAY
      foreach ($workflow['nodes'] as $node) {
        if (!is_array($node) || !isset($node['id'])) {
          continue;
        }
        
        $node_id = (string) $node['id']; // Convert to string WITHOUT hash
        
        $prompt_node = [
          'class_type' => $node['type'] ?? '', // Use 'type' not 'class_type'
          'inputs' => $this->extractNodeInputs($node),
        ];
        
        // Add metadata if present
        if (isset($node['_meta'])) {
          $prompt_node['_meta'] = $node['_meta'];
        }
        
        $prompt[$node_id] = $prompt_node;
      }
    } else if (is_array($workflow)) {
      // Already in dict/prompt format - verify structure
      $prompt = $workflow;
    }
    
    if (empty($prompt)) {
      throw new \Exception('No valid nodes found in workflow');
    }
    
    return $prompt;
  }

  /**
   * Extract inputs from workflow.json node format.
   * Handles both 'inputs' object and 'widgets_values' array.
   */
  protected function extractNodeInputs($node) {
    $inputs = [];
    
    // Standard inputs object
    if (isset($node['inputs']) && is_array($node['inputs'])) {
      // If it's already a dict of inputs, use it
      if (!isset($node['inputs'][0])) {
        // Dict format
        $inputs = $node['inputs'];
      } else {
        // If inputs is an array, it might be widget-based
        // Skip for now and try widgets_values
      }
    }
    
    // Widget values (for nodes that use widgets instead of typed inputs)
    if (isset($node['widgets_values']) && is_array($node['widgets_values'])) {
      // Try to convert widgets to inputs
      // This depends on the node type
    }
    
    return $inputs;
  }

  /**
   * Attempts to execute the workflow.
   *
   * @param array $workflow_data
   *   The workflow data to execute.
   *
   * @return array
   *   The execution result.
   */
  protected function attemptExecution(array $workflow_data) {
    $config = $this->configFactory->get('comfyui.settings');
    $api_endpoint = $config->get('api_endpoint');

    try {
      // Log the request for debugging
      $this->loggerFactory->get('comfyui')
        ->debug('Sending request to ComfyUI API: @data', [
          '@data' => json_encode($workflow_data),
        ]);

      $response = $this->httpClient->post($api_endpoint . '/prompt', 
        $this->buildRequestOptions([
          'json' => ['prompt' => $workflow_data],
        ])
      );

      $result = json_decode((string) $response->getBody(), TRUE);

      if (isset($result['prompt_id'])) {
        $this->loggerFactory->get('comfyui')
          ->info('Workflow execution started. Prompt ID: @prompt_id', [
            '@prompt_id' => $result['prompt_id'],
          ]);

        return [
          'success' => TRUE,
          'message' => 'Workflow execution started',
          'data' => ['prompt_id' => $result['prompt_id']],
        ];
      }

      return [
        'success' => FALSE,
        'message' => 'No prompt ID received',
        'error' => 'Invalid API response',
      ];
    }
    catch (ClientException $e) {
      $response_body = json_decode((string) $e->getResponse()->getBody(), TRUE);
      
      $this->loggerFactory->get('comfyui')
        ->error('API request failed: @error', [
          '@error' => $response_body['error']['message'] ?? $e->getMessage(),
        ]);

      return [
        'success' => FALSE,
        'error' => $response_body['error'] ?? $e->getMessage(),
        'node_errors' => $response_body['node_errors'] ?? [],
      ];
    }
  }

  /**
   * Applies custom input values to the workflow.
   *
   * @param array $workflow_data
   *   The workflow data.
   * @param array $input_values
   *   Array of input values to apply.
   *
   * @return array
   *   The modified workflow data.
   */
  protected function applyInputValues(array $workflow_data, array $input_values) {
    foreach ($input_values as $node_id => $inputs) {
      if (isset($workflow_data[$node_id])) {
        foreach ($inputs as $input_name => $value) {
          $workflow_data[$node_id]['inputs'][$input_name] = $value;
        }
      }
    }
    return $workflow_data;
  }

  /**
   * Check execution status using history endpoint
   */
  public function checkExecutionStatus($prompt_id) {
    $comfyui_endpoint = \Drupal::config('comfyui.settings')->get('api_endpoint');
    
    if (!$comfyui_endpoint) {
      throw new \Exception('ComfyUI API endpoint not configured');
    }
    
    try {
      $client = \Drupal::httpClient();
      
      // Build request options
      $request_options = [
        'timeout' => 10, // Longer timeout for history check
        'http_errors' => FALSE, // Don't throw on 404
      ];
      
      // Add authentication if login module is enabled
      if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
        $token_service = \Drupal::service('comfyui_login.token_service');
        $request_options = $token_service->buildRequestOptions($request_options);
      }
      
      // Check history endpoint (more reliable than queue)
      $history_url = $comfyui_endpoint . '/history/' . $prompt_id;
      $response = $client->get($history_url, $request_options);
      
      $status_code = $response->getStatusCode();
      $data = json_decode($response->getBody()->getContents(), TRUE);
      
      // If 200, execution is complete (exists in history)
      if ($status_code === 200 && !empty($data[$prompt_id])) {
        return [
          'success' => TRUE,
          'status' => 'completed',
          'prompt_id' => $prompt_id,
          'data' => $data[$prompt_id],
        ];
      }
      
      // If 404 or empty, execution not yet in history (still processing or queued)
      if ($status_code === 404 || empty($data[$prompt_id])) {
        // Try to get queue status to see if it's queued or running
        $queue_url = $comfyui_endpoint . '/queue';
        
        try {
          $queue_response = $client->get($queue_url, [
            'timeout' => 3, // Short timeout, we just want to know if queued
            'http_errors' => FALSE,
          ] + $request_options);
          
          $queue_data = json_decode($queue_response->getBody()->getContents(), TRUE);
          
          // Check if prompt is in queue (running or pending)
          $in_queue = FALSE;
          
          if (isset($queue_data['queue_running'])) {
            foreach ($queue_data['queue_running'] as $item) {
              if (isset($item[1]) && $item[1] === $prompt_id) {
                $in_queue = TRUE;
                break;
              }
            }
          }
          
          if (!$in_queue && isset($queue_data['queue_pending'])) {
            foreach ($queue_data['queue_pending'] as $item) {
              if (isset($item[1]) && $item[1] === $prompt_id) {
                $in_queue = TRUE;
                break;
              }
            }
          }
          
          return [
            'success' => TRUE,
            'status' => $in_queue ? 'processing' : 'waiting',
            'prompt_id' => $prompt_id,
            'data' => NULL,
          ];
          
        } catch (\Exception $e) {
          // Queue check failed, but not fatal - likely still processing
          return [
            'success' => TRUE,
            'status' => 'processing',
            'prompt_id' => $prompt_id,
            'data' => NULL,
          ];
        }
      }
      
      return [
        'success' => FALSE,
        'status' => 'error',
        'message' => 'Unexpected response from ComfyUI',
      ];
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Status check failed: @error', [
        '@error' => $e->getMessage()
      ]);
      
      return [
        'success' => FALSE,
        'status' => 'error',
        'message' => $e->getMessage(),
      ];
    }
  }

  /**
   * Gets formatted results for display based on field mappings.
   *
   * @param string $prompt_id
   *   The prompt ID to get results for.
   * @param int $workflow_id
   *   The workflow ID to get output mappings for.
   *
   * @return array
   *   Formatted results ready for display.
   */
  public function getFormattedResults($prompt_id, $workflow_id) {
    $results = [];
    
    // First check if execution is complete
    $status_result = $this->checkExecutionStatus($prompt_id);
    if (!$status_result['success']) {
      return ['error' => 'Could not retrieve execution status'];
    }
    
    if (!isset($status_result['data'][$prompt_id])) {
      return ['error' => 'Execution not found in history'];
    }
    
    $execution_data = $status_result['data'][$prompt_id];
    
    // Check if execution is complete
    $status = $execution_data['status']['status_str'] ?? 'unknown';
    if ($status !== 'success') {
      return ['error' => 'Execution not complete. Status: ' . $status];
    }
    
    $outputs = $execution_data['outputs'] ?? [];
    
    if (empty($outputs)) {
      return ['error' => 'No outputs found in execution results'];
    }
    
    // Log outputs for debugging
    $this->loggerFactory->get('comfyui')
      ->debug('Available outputs: @outputs', [
        '@outputs' => json_encode($outputs, JSON_PRETTY_PRINT),
      ]);
    
    // Load field mappings
    $mapping_storage = $this->entityTypeManager->getStorage('comfyui_node_mapping');
    $mappings = $mapping_storage->loadByProperties(['workflow_id' => $workflow_id]);
    
    if (empty($mappings)) {
      return ['error' => 'No field mappings found'];
    }
    
    $mapping = reset($mappings);
    $field_mappings = $mapping->getFieldMappings();
    $field_types = $mapping->getFieldTypes() ?: [];
    
    // Process each output field
    foreach ($field_mappings as $group_name => $group_fields) {
      foreach ($group_fields as $field_key => $node_mapping) {
        $field_type = $field_types[$group_name][$field_key] ?? 'input';
        
        // Only process output fields
        if ($field_type !== 'output') {
          continue;
        }
        
        // Parse node mapping (format: "node_id:output_name")
        if (strpos($node_mapping, ':') !== FALSE) {
          list($node_id, $output_name) = explode(':', $node_mapping, 2);
          
          if (isset($outputs[$node_id])) {
            $node_outputs = $outputs[$node_id];
            $results[$field_key] = $this->formatOutputData($node_outputs, $output_name);
          } else {
            $results[$field_key] = ['error' => "Node $node_id not found in outputs"];
          }
        }
      }
    }
    
    return $results;
  }

  /**
   * Debug method to see raw API response.
   */
  public function debugHistoryResponse($prompt_id) {
    try {
      $config = $this->configFactory->get('comfyui.settings');
      $api_endpoint = $config->get('api_endpoint');

      $response = $this->httpClient->get($api_endpoint . '/history/' . $prompt_id,
        $this->buildRequestOptions()
      );
      
      $result = json_decode((string) $response->getBody(), TRUE);
      
      // Log the full response for debugging
      $this->loggerFactory->get('comfyui')
        ->debug('History API Response: @data', [
          '@data' => json_encode($result, JSON_PRETTY_PRINT),
        ]);
      
      return $result;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('comfyui')
        ->error('History API failed: @error', ['@error' => $e->getMessage()]);
      return FALSE;
    }
  }

  /**
   * Transform input values using field mappings.
   *
   * @param array $input_values
   *   Raw input values from form submission.
   * @param \Drupal\comfyui\Entity\ComfyUIFieldMappingInterface $field_mapping
   *   The field mapping configuration.
   *
   * @return array
   *   Transformed values ready for ComfyUI.
   */
  protected function transformInputValuesWithFieldMapping(array $input_values, $field_mapping) {
    $transformed = [];
    $field_mappings = $field_mapping->getFieldMappings();

    foreach ($input_values as $field_name => $field_value) {
      if (!isset($field_mappings[$field_name])) {
        continue;
      }

      $mapping = $field_mappings[$field_name];
      
      // Only process input fields
      if ($mapping['direction'] !== 'input') {
        continue;
      }

      $node_id = $mapping['comfyui_node'] ?? null;
      $input_name = $mapping['comfyui_input'] ?? null;

      if ($node_id && $input_name) {
        $transformed_value = $this->transformSingleFieldValue($field_value, $mapping);
        
        if ($transformed_value !== null) {
          $transformed[$node_id][$input_name] = $transformed_value;
        }
      }
    }

    return $transformed;
  }

  /**
   * Transform a single field value based on its mapping configuration.
   *
   * @param mixed $field_value
   *   The raw field value.
   * @param array $mapping
   *   The field mapping configuration.
   *
   * @return mixed
   *   The transformed value.
   */
  protected function transformSingleFieldValue($field_value, array $mapping) {
    // **CHECK 1: Direct numeric ID (from media selector) - MUST BE FIRST**
    if ((is_numeric($field_value) && !is_array($field_value)) || 
        (is_string($field_value) && is_numeric(trim($field_value)))) {
      $target_id = (int) $field_value;
      
      try {
        $media = $this->entityTypeManager->getStorage('media')->load($target_id);
        
        if ($media && in_array($media->bundle(), ['comfyui_input', 'comfyui_generated'])) {
          // Generate tokenized URL for media
          $config = $this->configFactory->get('comfyui.settings');
          $expiration = $config->get('token_expiration') ?: 3600;
          
          $value = $this->mediaTokenService->generateTemporaryUrl($media, $expiration);
          
          if ($value) {
            return $value;
          }
        }
      } catch (\Exception $e) {
        $this->loggerFactory->get('comfyui')->warning(
          'Failed to convert media ID @id to URL: @error',
          ['@id' => $target_id, '@error' => $e->getMessage()]
        );
      }
    }
    
    // **CHECK 2: Handle different field value structures from Drupal**
    if (is_array($field_value)) {
      // Multi-value field or structured field
      if (isset($field_value[0]['value'])) {
        // Text field structure
        $value = $field_value[0]['value'];
      } elseif (isset($field_value[0]['target_id'])) {
        // Entity reference field - check if it's a media entity
        $target_id = $field_value[0]['target_id'];
        
        // Try to load as media entity
        try {
          $media = $this->entityTypeManager->getStorage('media')->load($target_id);
          
          if ($media && in_array($media->bundle(), ['comfyui_input', 'comfyui_generated'])) {
            // This is a ComfyUI media entity - generate tokenized URL
            $config = $this->configFactory->get('comfyui.settings');
            $expiration = $config->get('token_expiration') ?: 3600;
            
            $value = $this->mediaTokenService->generateTemporaryUrl($media, $expiration);
            
            if ($value) {
              $this->loggerFactory->get('comfyui')->info(
                'Generated tokenized URL for media @id (expires in @exp seconds)',
                ['@id' => $media->id(), '@exp' => $expiration]
              );
              return $value;
            } else {
              $this->loggerFactory->get('comfyui')->warning(
                'Failed to generate tokenized URL for media @id',
                ['@id' => $media->id()]
              );
              $value = null;
            }
          } else {
            // Not a media entity, just use the ID
            $value = $target_id;
          }
        } catch (\Exception $e) {
          // If loading fails, fall back to target_id
          $value = $target_id;
        }
      } elseif (isset($field_value[0]['uri'])) {
        // File/Image field - convert to URL
        $file = $this->entityTypeManager
          ->getStorage('file')
          ->load($field_value[0]['target_id']);
        if ($file) {
          $value = \Drupal::service('file_url_generator')->generateAbsoluteString($file->getFileUri());
        } else {
          $value = null;
        }
      } elseif (isset($field_value[0])) {
        // Other structured field
        $value = $field_value[0];
      } else {
        $value = $field_value;
      }
    } else {
      // Scalar value (string, int, etc.)
      $value = $field_value;
    }

    // Apply type-specific transformations based on mapping
    $field_type = $mapping['field_type'] ?? 'text';
    
    switch ($field_type) {
      case 'integer':
        return (int) $value;
      
      case 'float':
      case 'decimal':
        return (float) $value;
      
      case 'boolean':
        return (bool) $value;
      
      case 'list_string':
        // For select lists, return the selected option
        return is_array($value) ? $value[0] : $value;
      
      case 'image':
      case 'file':
        // Return URL for file fields
        return $value;
      
      default:
        // Text and other fields
        return (string) $value;
    }
  }

  /**
   * Apply field mapping inputs to workflow data.
   *
   * @param array $workflow_data
   *   The workflow data.
   * @param array $transformed_inputs
   *   Transformed input values organized by node.
   * @param \Drupal\comfyui\Entity\ComfyUIFieldMappingInterface $field_mapping
   *   The field mapping configuration.
   *
   * @return array
   *   The modified workflow data.
   */
  protected function applyFieldMappingInputs(array $workflow_data, array $transformed_inputs, $field_mapping) {
    foreach ($transformed_inputs as $node_id => $node_inputs) {
      if (isset($workflow_data[$node_id])) {
        foreach ($node_inputs as $input_name => $value) {
          $workflow_data[$node_id]['inputs'][$input_name] = $value;
          
          // Log the transformation
          $this->loggerFactory->get('comfyui')
            ->debug('Applied field mapping: Node @node, Input @input = @value', [
              '@node' => $node_id,
              '@input' => $input_name,
              '@value' => is_scalar($value) ? $value : json_encode($value),
            ]);
        }
      }
    }

    return $workflow_data;
  }

  /**
   * Format output data for a specific Drupal field.
   *
   * @param array $node_outputs
   *   Output data from ComfyUI node.
   * @param string $output_name
   *   The specific output name to extract.
   * @param array $mapping
   *   The field mapping configuration.
   *
   * @return array
   *   Formatted output data.
   */

  /**
   * Format image outputs for Drupal field display.
   */
  protected function formatImageOutputForField(array $images, $field_type) {
    $formatted_images = [];
    
    foreach ($images as $image) {
      // Use Drupal proxy for image URLs
      $proxy_url = \Drupal\Core\Url::fromRoute('comfyui.image_proxy', [
        'filename' => $image['filename']
      ], [
        'query' => [
          'subfolder' => $image['subfolder'] ?? '',
          'type' => $image['type'] ?? 'output',
        ],
      ])->toString();

      $formatted_images[] = [
        'type' => 'image',
        'url' => $proxy_url,
        'filename' => $image['filename'],
        'alt' => 'Generated image',
      ];
    }

    return $formatted_images;
  }

  /**
   * Format text outputs for Drupal field display.
   */
  protected function formatTextOutputForField($text_data, $field_type) {
    $content = is_array($text_data) ? implode("\n", $text_data) : $text_data;
    
    return [
      'type' => 'text',
      'content' => $content,
      'format' => 'plain_text',
    ];
  }

  /**
   * Format generic outputs for Drupal field display.
   */
  protected function formatGenericOutputForField($output_data, $field_type) {
    return [
      'type' => 'raw',
      'content' => is_scalar($output_data) ? $output_data : json_encode($output_data),
    ];
  }

  /**
   * Determine output type from ComfyUI result data.
   */
  protected function determineOutputType($output_data) {
    if (is_array($output_data) && !empty($output_data)) {
      $first_item = $output_data[0];
      
      if (is_array($first_item) && isset($first_item['filename'])) {
        return 'image';
      }
    }

    if (is_string($output_data)) {
      return 'text';
    }

    return 'data';
  }

  /**
   * Get execution results from ComfyUI history API.
   */
  protected function getExecutionHistory($prompt_id) {
      $comfyui_endpoint = \Drupal::config('comfyui.settings')->get('api_endpoint');
      
      if (!$comfyui_endpoint) {
          throw new \Exception('ComfyUI API endpoint not configured');
      }
      
      $history_url = $comfyui_endpoint . '/history/' . $prompt_id;
      
      try {
          $client = \Drupal::httpClient();
          
          // Build request options with authentication from login module if enabled
          $request_options = [
              'timeout' => 30,
              'headers' => ['Accept' => 'application/json'],
          ];
          
          if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
            $token_service = \Drupal::service('comfyui_login.token_service');
            $request_options = $token_service->buildRequestOptions($request_options);
          }
          
          $response = $client->get($history_url, $request_options);
          
          $data = json_decode($response->getBody()->getContents(), TRUE);
          
          if (empty($data) || !isset($data[$prompt_id])) {
              throw new \Exception("Execution not found in history: {$prompt_id}");
          }
          
          \Drupal::logger('comfyui')->info('Retrieved execution history for prompt @prompt_id', [
              '@prompt_id' => $prompt_id
          ]);
          
          return $data[$prompt_id];
          
      } catch (\Exception $e) {
          \Drupal::logger('comfyui')->error('Failed to get execution history: @error', [
              '@error' => $e->getMessage()
          ]);
          throw $e;
      }
  }

  /**
   * Extract outputs from ComfyUI history data.
   */
  protected function extractOutputsFromHistory($history_data) {
      if (!isset($history_data['outputs'])) {
          \Drupal::logger('comfyui')->warning('No outputs found in history data');
          return [];
      }
      
      $outputs = [];
      
      foreach ($history_data['outputs'] as $node_id => $node_outputs) {
          $outputs[$node_id] = [];
          
          foreach ($node_outputs as $output_name => $output_data) {
              if (is_array($output_data)) {
                  // Handle file-based outputs (images, videos)
                  $processed_outputs = [];
                  foreach ($output_data as $item) {
                      if (isset($item['filename'])) {
                          $processed_outputs[] = [
                              'filename' => $item['filename'],
                              'subfolder' => $item['subfolder'] ?? '',
                              'type' => $item['type'] ?? 'output',
                          ];
                      } else {
                          $processed_outputs[] = $item;
                      }
                  }
                  $outputs[$node_id][$output_name] = $processed_outputs;
              } else {
                  // Handle data-based outputs (text, numbers)
                  $outputs[$node_id][$output_name] = $output_data;
              }
          }
      }
      
      \Drupal::logger('comfyui')->info('Extracted outputs from history: @outputs', [
          '@outputs' => print_r(array_keys($outputs), TRUE)
      ]);
      
      return $outputs;
  }

  /**
   * Execute workflow using configuration instead of separate field mapping entity.
   *
   * @param int $workflow_id
   *   The workflow entity ID.
   * @param array $input_values
   *   Array of field values from the form.
   * @param string $view_mode
   *   The configuration type ('form', 'api', 'headless', 'batch').
   *
   * @return array
   *   The execution result array.
   */
  public function executeWorkflowWithViewMode($workflow_id, array $input_values, $view_mode = 'form') {
    try {
      // Load workflow
      $workflow = $this->entityTypeManager->getStorage('comfyui_workflow')->load($workflow_id);
      if (!$workflow) {
        throw new \InvalidArgumentException("Workflow ID {$workflow_id} not found");
      }

      // Get field mappings (shared across all view modes)
      $field_mappings = $workflow->getFieldMappings();
      
      // Load and parse the workflow
      $workflow_data = $this->loadWorkflowData($workflow_id);
      
      // Transform input values using field mappings
      $transformed_inputs = $this->transformInputValues($input_values, $field_mappings);
      
      // Apply transformed values to workflow
      if (!empty($transformed_inputs)) {
        $workflow_data = $this->applyMappingArrayInputs($workflow_data, $transformed_inputs, $field_mappings);
      }

      // **APPLY BYPASS MAPPINGS - Rewire and remove bypassed nodes**
      $workflow_data = $this->rewriteBypassedNodes($workflow_data, $input_values, $workflow);

      // **INJECT SECURE FILENAME PREFIX INTO SAVEIMAGE NODES**
      $secure_prefix = bin2hex(random_bytes(16)); // 32-char random hash
      $workflow_data = $this->injectSecureFilenamePrefix($workflow_data, $secure_prefix);

      // Execute the workflow
      $execution_result = $this->attemptExecution($workflow_data);
      
      // **STORE PREFIX MAPPING FOR LATER RETRIEVAL**
      if ($execution_result['success'] && isset($execution_result['data']['prompt_id'])) {
        $prompt_id = $execution_result['data']['prompt_id'];
        \Drupal::state()->set('comfyui_prompt_' . $prompt_id . '_prefix', $secure_prefix);
        \Drupal::logger('comfyui')->info('Stored filename prefix for prompt @prompt: @prefix', [
          '@prompt' => $prompt_id,
          '@prefix' => $secure_prefix,
        ]);
      }
      
      return $execution_result;

    } catch (\Exception $e) {
      $this->loggerFactory->get('comfyui')
        ->error('Workflow execution with view mode failed: @error', [
          '@error' => $e->getMessage(),
        ]);
      return [
        'success' => FALSE,
        'message' => 'Workflow execution failed',
        'error' => $e->getMessage(),
      ];
    }
  }

  /**
   * Get formatted results using workflow configuration.
   *
   * @param string $prompt_id
   *   The prompt ID to get results for.
   * @param int $workflow_id
   *   The workflow ID to get output mappings for.
   * @param string $view_mode
   *   The view mode.
   *
   * @return array
   *   Formatted results ready for display.
   */
  public function getFormattedResultsWithViewMode($prompt_id, $workflow_id, $view_mode = 'form') {
    try {
      // Load workflow
      $workflow = $this->entityTypeManager->getStorage('comfyui_workflow')->load($workflow_id);
      if (!$workflow) {
        throw new \Exception("Workflow ID {$workflow_id} not found");
      }

      // Get execution results from ComfyUI history
      $history_data = $this->getExecutionHistory($prompt_id);
      $outputs = $this->extractOutputsFromHistory($history_data);
      
      if (empty($outputs)) {
        throw new \Exception("No outputs found in execution history for prompt: {$prompt_id}");
      }
      
      // Get field mappings (shared across all view modes)
      $field_mappings = $workflow->getFieldMappings();
      $results = [];
      
      // Process each mapped output field
      foreach ($field_mappings as $field_name => $mapping) {
        if ($mapping['direction'] !== 'output') {
          continue; // Skip input fields
        }
        
        $node_id = $mapping['comfyui_node'];
        $output_name = $mapping['comfyui_output'];
        
        \Drupal::logger('comfyui')->info('Processing output mapping: @field -> node @node, output @output', [
          '@field' => $field_name,
          '@node' => $node_id,
          '@output' => $output_name,
        ]);
        
        if (isset($outputs[$node_id][$output_name])) {
          $output_data = $outputs[$node_id][$output_name];
          
          // For file-based outputs (arrays), take the first item
          if (is_array($output_data) && !empty($output_data) && isset($output_data[0]['filename'])) {
            $results[$field_name] = $output_data[0]; // First image/file
          } else {
            // Data-based outputs (text, numbers)
            $results[$field_name] = $output_data;
          }
        } else {
          $results[$field_name] = [
            'error' => "Output '{$output_name}' not found in node {$node_id}"
          ];
        }
      }
      
      return $results;
      
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->error('Failed to get formatted results: @error', [
        '@error' => $e->getMessage()
      ]);
      throw $e;
    }
  }

  /**
   * Transform input values using configuration field mappings.
   *
   * @param array $input_values
   *   Raw input values from form submission.
   * @param array $field_mappings
   *   The field mappings from workflow configuration.
   *
   * @return array
   *   Transformed values ready for ComfyUI.
   */
  protected function transformInputValues(array $input_values, array $field_mappings) {
    $transformed = [];

    foreach ($input_values as $field_name => $field_value) {
      if (!isset($field_mappings[$field_name])) {
        continue;
      }

      $mapping = $field_mappings[$field_name];
      
      // Only process input fields
      if ($mapping['direction'] !== 'input') {
        continue;
      }

      $node_id = $mapping['comfyui_node'] ?? null;
      $input_name = $mapping['comfyui_input'] ?? null;

      if ($node_id && $input_name) {
        $transformed_value = $this->transformSingleFieldValue($field_value, $mapping);
        
        if ($transformed_value !== null) {
          $transformed[$node_id][$input_name] = $transformed_value;
        }
      }
    }

    return $transformed;
  }

  /**
   * Apply configuration inputs to workflow data.
   *
   * @param array $workflow_data
   *   The workflow data.
   * @param array $transformed_inputs
   *   Transformed input values organized by node.
   * @param array $field_mappings
   *   The field mappings configuration.
   *
   * @return array
   *   The modified workflow data.
   */
  protected function applyMappingArrayInputs(array $workflow_data, array $transformed_inputs, array $field_mappings) {
    foreach ($transformed_inputs as $node_id => $node_inputs) {
      if (isset($workflow_data[$node_id])) {
        foreach ($node_inputs as $input_name => $value) {
          $workflow_data[$node_id]['inputs'][$input_name] = $value;
          
          // Log the transformation
          $this->loggerFactory->get('comfyui')
            ->debug('Applied configuration mapping: Node @node, Input @input = @value', [
              '@node' => $node_id,
              '@input' => $input_name,
              '@value' => is_scalar($value) ? $value : json_encode($value),
            ]);
        }
      }
    }

    return $workflow_data;
  }



  /**
   * Legacy method for backward compatibility during transition.
   * 
   * @deprecated Use executeWorkflowWithViewMode() instead.
   */
  public function executeWorkflowWithFieldMapping($workflow_id, array $input_values, $field_mapping) {
    // For backward compatibility during transition
    \Drupal::logger('comfyui')->warning('executeWorkflowWithFieldMapping is deprecated. Use executeWorkflowWithViewMode instead.');
    return $this->executeWorkflowWithViewMode($workflow_id, $input_values, 'form');
  }

  /**
   * Legacy method for backward compatibility during transition.
   * 
   * @deprecated Use getFormattedResultsWithViewMode() instead.
   */
  public function getFormattedResultsWithFieldMapping($prompt_id, $field_mapping) {
    // For backward compatibility during transition
    \Drupal::logger('comfyui')->warning('getFormattedResultsWithFieldMapping is deprecated. Use getFormattedResultsWithViewMode instead.');
    
    // We need to get the workflow ID from somewhere - this is a limitation of the legacy method
    // In practice, this method should not be called anymore
    throw new \Exception('Legacy method getFormattedResultsWithFieldMapping cannot determine workflow ID. Use getFormattedResultsWithViewMode instead.');
  }

  /**
   * Legacy method for backward compatibility during transition.
   * 
   * @deprecated Use executeWorkflowWithViewMode() instead.
   */
  public function executeWorkflowWithConfiguration($workflow_id, array $input_values, $view_mode = 'form') {
    \Drupal::logger('comfyui')->warning('executeWorkflowWithConfiguration is deprecated. Use executeWorkflowWithViewMode instead.');
    return $this->executeWorkflowWithViewMode($workflow_id, $input_values, $view_mode);
  }

  /**
   * Legacy method for backward compatibility during transition.
   * 
   * @deprecated Use getFormattedResultsWithViewMode() instead.
   */
  public function getFormattedResultsWithConfiguration($prompt_id, $workflow_id, $view_mode = 'form') {
    \Drupal::logger('comfyui')->warning('getFormattedResultsWithConfiguration is deprecated. Use getFormattedResultsWithViewMode instead.');
    return $this->getFormattedResultsWithViewMode($prompt_id, $workflow_id, $view_mode);
  }

  /**
   * Save a generated image as a media entity.
   *
   * @param string $filename
   *   The filename from ComfyUI output.
   * @param string $subfolder
   *   The subfolder from ComfyUI output.
   * @param string $type
   *   The type from ComfyUI output.
   * @param int $workflow_id
   *   The workflow ID that generated this image.
   * @param int|null $source_media_id
   *   Optional source media ID if this was generated from another image.
   * @param string|null $prompt
   *   Optional prompt text used for generation.
   * @param array $workflow_params
   *   Optional workflow parameters used.
   * @param string|null $prompt_id
   *   The ComfyUI prompt ID.
   *
   * @return \Drupal\media\MediaInterface|null
   *   The created media entity, or NULL on failure.
   */
  public function saveGeneratedImageAsMedia($filename, $subfolder, $type, $workflow_id, $source_media_id = NULL, $prompt = NULL, array $workflow_params = [], $prompt_id = NULL) {
    try {
      $config = $this->configFactory->get('comfyui.settings');
      $api_endpoint = $config->get('api_endpoint');
      
      // Build ComfyUI image URL
      $image_url = $api_endpoint . '/view?' . http_build_query([
        'filename' => $filename,
        'subfolder' => $subfolder,
        'type' => $type,
      ]);
      
      // Download image from ComfyUI
      $request_options = [
        'timeout' => 30,
      ];
      
      // Add authentication if login module is enabled
      if (\Drupal::moduleHandler()->moduleExists('comfyui_login')) {
        $token_service = \Drupal::service('comfyui_login.token_service');
        $request_options = $token_service->buildRequestOptions($request_options);
      }
      
      $response = $this->httpClient->get($image_url, $request_options);
      $image_data = $response->getBody()->getContents();
      
      // Determine file extension
      $extension = pathinfo($filename, PATHINFO_EXTENSION) ?: 'png';

      // Create unique filename with random hash (security: prevent enumeration)
      $random_hash = bin2hex(random_bytes(16)); // 32-char hex string
      $new_filename = $random_hash . '.' . $extension;

      // Store filename mapping for audit trail
      $this->storeFilenameMapping($filename, $random_hash, $prompt_id, $workflow_id);
      
      // Get directory from helper - uses comfyui_file_scheme setting
      $helper = \Drupal::service('comfyui.file_scheme_helper');
      $directory = $helper->getGeneratedDirectory();
      
      // Prepare directory
      $this->fileSystem->prepareDirectory(
        $directory,
        \Drupal\Core\File\FileSystemInterface::CREATE_DIRECTORY
      );
      
      // Build the full file URI
      $file_uri = $directory . '/' . $new_filename;
      
      // Save file using file_system service
      $this->fileSystem->saveData(
        $image_data,
        $file_uri,
        \Drupal\Core\File\FileSystemInterface::EXISTS_RENAME
      );
      
      // Get current user
      $current_user = \Drupal::currentUser();
      
      // Create file entity
      $file = $this->entityTypeManager->getStorage('file')->create([
        'uri' => $file_uri,
        'uid' => $current_user->id(),
      ]);
      $file->setPermanent();
      $file->save();
      
      if (!$file) {
        throw new \Exception('Failed to save image file');
      }
      
      // Create media entity
      $media = $this->entityTypeManager->getStorage('media')->create([
        'bundle' => 'comfyui_generated',
        'uid' => $current_user->id(),
        'name' => 'Generated: ' . date('Y-m-d H:i:s'),
        'field_media_image' => [
          'target_id' => $file->id(),
          'alt' => $prompt ? substr($prompt, 0, 100) : 'Generated image',
        ],
        'field_comfyui_workflow' => $workflow_id,
        'field_source_image' => $source_media_id,
        'field_prompt' => $prompt,
        'field_workflow_params' => !empty($workflow_params) ? json_encode($workflow_params) : NULL,
        'field_comfyui_prompt_id' => $prompt_id,
      ]);
      
      $media->save();
      
      $this->loggerFactory->get('comfyui')->info(
        'Saved generated image as media entity @id (file: @file)',
        ['@id' => $media->id(), '@file' => $filename]
      );
      
      // Generate tokenized URL for display
      $token_service = \Drupal::service('comfyui.media_token');
      $media->_tokenized_url = $token_service->generateTemporaryUrl($media, 3600);
      
      return $media;
      
    } catch (\Exception $e) {
      $this->loggerFactory->get('comfyui')->error(
        'Failed to save generated image as media: @error',
        ['@error' => $e->getMessage()]
      );
      return NULL;
    }
  }

  /**
   * Store mapping of ComfyUI filename to Drupal hash for security.
   * Prevents enumeration attacks by hiding sequential filenames.
   */
  protected function storeFilenameMapping($comfyui_filename, $drupal_hash, $prompt_id, $workflow_id) {
    try {
      // Store in cache for quick lookup
      $mapping_data = [
        'comfyui_filename' => $comfyui_filename,
        'drupal_hash' => $drupal_hash,
        'prompt_id' => $prompt_id,
        'workflow_id' => $workflow_id,
        'timestamp' => time(),
        'user_id' => \Drupal::currentUser()->id(),
      ];
      
      // Cache key combines both names for bidirectional lookup
      $cache_key = 'comfyui_filename_map_' . md5($comfyui_filename . $prompt_id);
      \Drupal::cache()->set($cache_key, $mapping_data, time() + (7 * 24 * 60 * 60)); // 7 days
      
      // Also store reverse mapping
      $reverse_cache_key = 'comfyui_hash_map_' . $drupal_hash;
      \Drupal::cache()->set($reverse_cache_key, $mapping_data, time() + (7 * 24 * 60 * 60));
      
      \Drupal::logger('comfyui')->debug('Stored filename mapping: @comfyui -> @drupal for prompt @prompt', [
        '@comfyui' => $comfyui_filename,
        '@drupal' => $drupal_hash,
        '@prompt' => $prompt_id,
      ]);
    } catch (\Exception $e) {
      \Drupal::logger('comfyui')->warning('Failed to store filename mapping: @error', [
        '@error' => $e->getMessage()
      ]);
    }
  }

  /**
   * Get original ComfyUI filename from Drupal hash.
   */
  public function getComfyUIFilenameFromHash($drupal_hash) {
    $cache_key = 'comfyui_hash_map_' . $drupal_hash;
    $cached = \Drupal::cache()->get($cache_key);
    
    if ($cached && isset($cached->data['comfyui_filename'])) {
      return $cached->data['comfyui_filename'];
    }
    
    return NULL;
  }

  /**
   * Inject secure random filename prefix into all SaveImage nodes.
   * This prevents sequential filename enumeration attacks.
   */
  protected function injectSecureFilenamePrefix(array $workflow_data, $secure_prefix) {
    foreach ($workflow_data as $node_id => &$node) {
      // Check if this is a SaveImage-like node
      if (isset($node['class_type'])) {
        $class_type = $node['class_type'];
        
        // Match any node with "Save" in the name (SaveImage, SaveImage_NoWorkflow, etc.)
        if (stripos($class_type, 'Save') !== FALSE && isset($node['inputs'])) {
          // Override filename_prefix input with secure hash
          if (isset($node['inputs']['filename_prefix'])) {
            $node['inputs']['filename_prefix'] = $secure_prefix;
            
            \Drupal::logger('comfyui')->debug('Injected secure prefix into node @node (@type)', [
              '@node' => $node_id,
              '@type' => $class_type,
            ]);
          }
        }
      }
    }
    
    return $workflow_data;
  }

  /**
   * Rewrite workflow by removing bypassed nodes and connecting their inputs to outputs.
   * 
   * Simple logic:
   * - For each bypassed node, find what inputs feed INTO it
   * - Find what outputs feed FROM it
   * - Connect anything that references that bypassed node's output
   *   to instead use one of its inputs
   */
  protected function rewriteBypassedNodes(array $workflow_data, array $input_values, $workflow) {
    try {
      $workflow_json = $workflow->getWorkflowJson();
      if (!$workflow_json) {
        return $workflow_data;
      }

      $workflow_full = json_decode($workflow_json, TRUE);
      $bypass_directives = $this->bypassMappingService->buildBypassDirectives(
        $workflow,
        $input_values,
        $workflow_full
      );

      if (empty($bypass_directives)) {
        return $workflow_data;
      }

      $rewired = $workflow_data;
      $bypass_set = array_map('strval', $bypass_directives);
      
      // For each bypassed node, map its output → one of its inputs
      $output_redirects = [];
      
      foreach ($bypass_directives as $node_id) {
        $node_id_str = (string) $node_id;
        
        if (!isset($rewired[$node_id_str])) {
          continue;
        }

        $node = $rewired[$node_id_str];
        
        // Get all inputs that feed into this bypassed node
        $node_inputs = [];
        if (isset($node['inputs']) && is_array($node['inputs'])) {
          foreach ($node['inputs'] as $input_name => $input_value) {
            if (is_array($input_value) && isset($input_value[0])) {
              $node_inputs[] = $input_value;
            }
          }
        }

        // If this bypassed node receives inputs, map its outputs to those inputs
        if (!empty($node_inputs)) {
          $output_redirects[$node_id_str] = $node_inputs[0];
        }
      }

      // Rewrite references - do this repeatedly until stable
      // (in case of cascading references)
      $max_iterations = 10;
      $iteration = 0;
      $total_updated = 0;
      
      while ($iteration < $max_iterations) {
        $updated_this_pass = 0;
        
        foreach ($rewired as $node_id => &$node) {
          if (!isset($node['inputs']) || !is_array($node['inputs'])) {
            continue;
          }

          foreach ($node['inputs'] as $input_name => &$input) {
            if (is_array($input) && isset($input[0])) {
              $referenced_node = (string) $input[0];
              
              // If this references a bypassed/redirected node
              if (isset($output_redirects[$referenced_node])) {
                $node['inputs'][$input_name] = $output_redirects[$referenced_node];
                $updated_this_pass++;
              }
            }
          }
        }

        $total_updated += $updated_this_pass;
        
        // If nothing changed this pass, we're done
        if ($updated_this_pass === 0) {
          break;
        }
        
        $iteration++;
      }

      // Remove bypassed nodes
      $removed_count = 0;
      foreach ($bypass_directives as $node_id) {
        $node_id_str = (string) $node_id;
        if (isset($rewired[$node_id_str])) {
          unset($rewired[$node_id_str]);
          $removed_count++;
        }
      }

      if ($removed_count > 0) {
        $this->loggerFactory->get('comfyui')->info(
          'Node bypass rewiring: removed @removed nodes, updated @updated references in @iterations passes',
          ['@removed' => $removed_count, '@updated' => $total_updated, '@iterations' => $iteration]
        );
      }

      return $rewired;

    } catch (\Exception $e) {
      $this->loggerFactory->get('comfyui')->error(
        'Node bypass rewiring failed: @error',
        ['@error' => $e->getMessage()]
      );
      return $workflow_data;
    }
  }

  /**
   * Save original workflow API format before converting to full format.
   * Used for output extraction since outputs come in API format from history.
   */
  protected function getWorkflowApiFormat($workflow_id) {
    $workflow = $this->entityTypeManager
      ->getStorage('comfyui_workflow')
      ->load($workflow_id);

    if (!$workflow) {
      throw new \InvalidArgumentException("Workflow ID {$workflow_id} not found");
    }

    $workflow_file = $workflow->get('workflow_api')->entity;
    if (!$workflow_file) {
      throw new \InvalidArgumentException("Workflow API JSON file not found for workflow ID {$workflow_id}");
    }

    $file_uri = $workflow_file->getFileUri();
    $real_path = $this->fileSystem->realpath($file_uri);

    return $this->workflowParser->parseWorkflowJson($real_path);
  }

}