<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for parsing and validating ComfyUI workflow files.
 */
class ComfyUIWorkflowParser {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a ComfyUIWorkflowParser object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory) {
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Parses and validates a workflow JSON file.
   *
   * @param string $file_uri
   *   The URI of the workflow JSON file.
   *
   * @return array
   *   The parsed workflow data.
   *
   * @throws \InvalidArgumentException
   *   If the JSON is invalid or missing required structure.
   */
  public function parseWorkflowJson($file_uri) {
    if (!file_exists($file_uri)) {
      throw new \InvalidArgumentException('Workflow file not found');
    }

    $json_content = file_get_contents($file_uri);
    $data = json_decode($json_content, TRUE);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
      $this->loggerFactory->get('comfyui')
        ->error('Invalid JSON format in workflow file: @error', [
          '@error' => json_last_error_msg(),
        ]);
      throw new \InvalidArgumentException('Invalid JSON format in workflow file');
    }

    // Log the parsed data for debugging
    $this->loggerFactory->get('comfyui')
      ->debug('Parsed workflow JSON: @json', [
        '@json' => json_encode($data),
      ]);
    
    return $data;
  }

  /**
   * Validates field mappings against workflow nodes.
   *
   * @param array $workflow_data
   *   The workflow data.
   * @param array $field_mappings
   *   The field mappings to validate.
   *
   * @return bool
   *   TRUE if mappings are valid, FALSE otherwise.
   */
  public function validateFieldMappings(array $workflow_data, array $field_mappings) {
    foreach ($field_mappings as $field_name => $mapping) {
      // Validate node exists
      if (!isset($mapping['node_id']) || !isset($workflow_data[$mapping['node_id']])) {
        $this->loggerFactory->get('comfyui')
          ->warning('Invalid node ID in field mapping: @field', [
            '@field' => $field_name,
          ]);
        return FALSE;
      }

      // Validate input exists in node
      $node = $workflow_data[$mapping['node_id']];
      if (!isset($node['inputs'][$mapping['input']])) {
        $this->loggerFactory->get('comfyui')
          ->warning('Invalid input @input for node @node in field mapping: @field', [
            '@input' => $mapping['input'],
            '@node' => $mapping['node_id'],
            '@field' => $field_name,
          ]);
        return FALSE;
      }
    }
    
    return TRUE;
  }

  /**
   * Discover input fields from API format data.
   *
   * @param array $api_data
   *   The decoded API format JSON data.
   *
   * @return array
   *   Array of discovered input fields with keys: node_id, input_name, type, default_value.
   */
  public function discoverInputFields(array $api_data) {
    $inputs = [];
    
    foreach ($api_data as $node_id => $node) {
      if (empty($node['inputs'])) {
        continue;
      }
      
      foreach ($node['inputs'] as $input_name => $input_value) {
        // Skip array inputs that are connections to other nodes
        if (is_array($input_value) && isset($input_value[0])) {
          continue;
        }
        
        $type = $this->determineInputType($input_value);
        $inputs[] = [
          'node_id' => $node_id,
          'input_name' => $input_name,
          'type' => $type,
          'default_value' => $input_value,
          'class_type' => $node['class_type'] ?? 'unknown',
        ];
      }
    }
    
    return $inputs;
  }

  /**
   * Discover output fields from workflow data.
   *
   * @param array $workflow_data
   *   The decoded workflow JSON data.
   *
   * @return array
   *   Array of discovered output fields with keys: node_id, output_name, type, source.
   */
  public function discoverOutputFields(array $workflow_data) {
    $outputs = [];
    
    if (empty($workflow_data['nodes'])) {
      return $outputs;
    }

    // Common patterns for output nodes (regex patterns)
    $output_indicators = [
      '/save/i',          // SaveImage, SaveImageWebp, etc.
      '/preview/i',       // PreviewImage, ImagePreview, etc.
      '/display/i',       // Display nodes
      '/show/i',          // ShowText, ShowImage, etc.
      '/output/i',        // Output nodes
      '/export/i',        // Export nodes
    ];

    foreach ($workflow_data['nodes'] as $node) {
      $node_id = $node['id'] ?? null;
      $class_type = $node['type'] ?? 'unknown';
      $title = $node['title'] ?? '';
      
      if (!$node_id) {
        continue;
      }

      // Check if this node is likely an output node
      $is_output_node = FALSE;
      foreach ($output_indicators as $pattern) {
        if (preg_match($pattern, $class_type) || preg_match($pattern, $title)) {
          $is_output_node = TRUE;
          break;
        }
      }

      if (!$is_output_node) {
        continue;
      }

      // For output nodes, try to detect common output field names
      $output_fields = $this->detectOutputFields($class_type);
      
      if (!empty($output_fields)) {
        foreach ($output_fields as $output_name) {
          $outputs[] = [
            'node_id' => $node_id,
            'output_name' => $output_name,
            'type' => $output_name,
            'source' => 'node',
            'class_type' => $class_type,
            'title' => $title,
          ];
        }
      } else {
        // Fallback: assume common output names for unknown output nodes
        $default_outputs = ['images', 'image', 'text', 'output'];
        foreach ($default_outputs as $output_name) {
          $outputs[] = [
            'node_id' => $node_id,
            'output_name' => $output_name,
            'type' => $output_name,
            'source' => 'node',
            'class_type' => $class_type,
            'title' => $title,
          ];
        }
      }
    }
    
    return $outputs;
  }

  /**
   * Detect output field names for different node types.
   */
  protected function detectOutputFields($class_type) {
    // Map of class type patterns to their output fields
    $output_field_map = [
      'SaveImage' => ['images'],
      'PreviewImage' => ['images'],
      'ShowText' => ['text'],
      'StringFunction' => ['text'],
      'ImageSave' => ['images'],
      'VHS_VideoCombine' => ['video'],
      'SaveImage_NoWorkflow' => ['images'], // Hangover node
      'SaveImageWebp' => ['images'],
      'SaveAnimatedPNG' => ['images'],
      'SaveLatent' => ['latent'],
    ];

    // Direct match first
    if (isset($output_field_map[$class_type])) {
      return $output_field_map[$class_type];
    }

    // Pattern matching for custom nodes
    if (stripos($class_type, 'save') !== FALSE || stripos($class_type, 'image') !== FALSE) {
      return ['images', 'image'];
    }
    if (stripos($class_type, 'video') !== FALSE) {
      return ['video'];
    }
    if (stripos($class_type, 'text') !== FALSE || stripos($class_type, 'show') !== FALSE) {
      return ['text'];
    }

    return [];
  }

  /**
   * Determine the input type based on the value.
   *
   * @param mixed $value
   *   The input value.
   *
   * @return string
   *   The determined type.
   */
  protected function determineInputType($value) {
    if (is_string($value)) {
      return 'text';
    }
    if (is_int($value)) {
      return 'integer';
    }
    if (is_float($value)) {
      return 'float';
    }
    if (is_bool($value)) {
      return 'boolean';
    }
    if (is_array($value)) {
      return 'array';
    }
    
    return 'unknown';
  }

  /**
   * Extracts unique node types from workflow data.
   *
   * @param array $workflow_data
   *   The decoded workflow JSON data.
   *
   * @return array
   *   Array of unique node class types found in the workflow.
   */
  public function extractUniqueNodeTypes(array $workflow_data) {
    $node_types = [];
    
    foreach ($workflow_data as $node_id => $node) {
      if (isset($node['class_type'])) {
        $node_types[$node['class_type']] = $node['class_type'];
      }
    }
    
    return array_values($node_types);
  }

  /**
   * Enhanced field discovery with API options integration.
   *
   * @param array $api_data
   *   The decoded API format JSON data.
   * @param \Drupal\comfyui\Service\ComfyUIApiOptionsService $api_service
   *   The API options service.
   *
   * @return array
   *   Enhanced array of discovered input fields with available options.
   */
  public function discoverInputFieldsWithOptions(array $api_data, $api_service = NULL) {
    $inputs = $this->discoverInputFields($api_data);
    
    if (!$api_service) {
      return $inputs;
    }

    // Extract unique node types
    $node_types = $this->extractUniqueNodeTypes($api_data);
    
    // Fetch object info for all node types
    $node_definitions = $api_service->getMultipleObjectInfo($node_types);
    
    // Enhance inputs with available options
    foreach ($inputs as &$input) {
      $class_type = $input['class_type'];
      $input_name = $input['input_name'];
      
      if (isset($node_definitions[$class_type])) {
        $options = $api_service->extractInputOptions($node_definitions[$class_type], $input_name);
        
        if (!empty($options)) {
          $input['available_options'] = $options;
          $input['option_source'] = 'object_info';
          $input['has_dynamic_options'] = TRUE;
        }
      }
    }
    
    return $inputs;
  }

  /**
   * Discovers node definitions for a complete workflow.
   *
   * @param array $api_data
   *   The decoded API format JSON data.
   * @param \Drupal\comfyui\Service\ComfyUIApiOptionsService $api_service
   *   The API options service.
   *
   * @return array
   *   Complete discovery data including inputs, outputs, and node definitions.
   */
  public function discoverCompleteWorkflowDefinitions(array $api_data, array $workflow_data, $api_service = NULL) {
    // Get enhanced inputs
    $inputs = $this->discoverInputFieldsWithOptions($api_data, $api_service);
    
    // Get outputs (existing functionality)
    $outputs = $this->discoverOutputFields($workflow_data);
    
    // Get node definitions
    $node_definitions = [];
    if ($api_service) {
      $node_types = $this->extractUniqueNodeTypes($api_data);
      $node_definitions = $api_service->getMultipleObjectInfo($node_types);
    }
    
    return [
      'inputs' => $inputs,
      'outputs' => $outputs,
      'node_definitions' => $node_definitions,
      'node_types' => $this->extractUniqueNodeTypes($api_data),
    ];
  }

}