<?php

namespace Drupal\comfyui\Service;

use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for managing workflow execution queue.
 */
class ComfyUIWorkflowQueueManager {

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a WorkflowQueueManager object.
   *
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    QueueFactory $queue_factory,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->queueFactory = $queue_factory;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Queues a workflow for execution.
   *
   * @param int $workflow_id
   *   The workflow entity ID.
   * @param array $field_values
   *   The field values to apply.
   * @param callable|null $callback
   *   Optional callback to execute after workflow completion.
   *
   * @return bool
   *   TRUE if queued successfully, FALSE otherwise.
   */
  public function queueWorkflow($workflow_id, array $field_values = [], callable $callback = NULL) {
    try {
      $queue = $this->queueFactory->get('comfyui_workflow_executor');
      $config = $this->configFactory->get('comfyui.settings');

      // Check queue size
      if ($queue->numberOfItems() >= $config->get('queue_size')) {
        throw new \Exception('Queue is full');
      }

      // Prepare queue item
      $item = [
        'workflow_id' => $workflow_id,
        'field_values' => $field_values,
      ];

      if ($callback) {
        $item['callback'] = $callback;
      }

      // Add to queue
      if (!$queue->createItem($item)) {
        throw new \Exception('Failed to add item to queue');
      }

      $this->loggerFactory->get('comfyui')
        ->info('Workflow @id queued for execution', [
          '@id' => $workflow_id,
        ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('comfyui')
        ->error('Failed to queue workflow: @error', [
          '@error' => $e->getMessage(),
        ]);
      return FALSE;
    }
  }

  /**
   * Gets the current queue status.
   *
   * @return array
   *   An array containing queue statistics.
   */
  public function getQueueStatus() {
    $queue = $this->queueFactory->get('comfyui_workflow_executor');
    $config = $this->configFactory->get('comfyui.settings');

    return [
      'items' => $queue->numberOfItems(),
      'capacity' => $config->get('queue_size'),
      'available' => $config->get('queue_size') - $queue->numberOfItems(),
    ];
  }

  /**
   * Clears the workflow execution queue.
   *
   * @return bool
   *   TRUE if cleared successfully, FALSE otherwise.
   */
  public function clearQueue() {
    try {
      $queue = $this->queueFactory->get('comfyui_workflow_executor');
      $queue->deleteQueue();

      $this->loggerFactory->get('comfyui')
        ->info('Workflow execution queue cleared');

      return TRUE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('comfyui')
        ->error('Failed to clear queue: @error', [
          '@error' => $e->getMessage(),
        ]);
      return FALSE;
    }
  }

  /**
   * Queues a workflow for execution with configuration support.
   *
   * @param int $workflow_id
   *   The workflow entity ID.
   * @param array $field_values
   *   The field values to apply.
   * @param string $view_mode
   *   The configuration type ('form', 'api', 'headless', 'batch').
   * @param callable|null $callback
   *   Optional callback to execute after workflow completion.
   *
   * @return bool
   *   TRUE if queued successfully, FALSE otherwise.
   */
  public function queueWorkflowWithViewMode($workflow_id, array $field_values = [], $view_mode = 'form', callable $callback = NULL) {
    try {
      $queue = $this->queueFactory->get('comfyui_workflow_executor');
      $config = $this->configFactory->get('comfyui.settings');

      // Check queue size
      if ($queue->numberOfItems() >= $config->get('queue_size')) {
        throw new \Exception('Queue is full');
      }

      // Prepare queue item with configuration support
      $item = [
        'workflow_id' => $workflow_id,
        'field_values' => $field_values,
        'view_mode' => $view_mode,
      ];

      if ($callback) {
        $item['callback'] = $callback;
      }

      // Add to queue
      if (!$queue->createItem($item)) {
        throw new \Exception('Failed to add item to queue');
      }

      $this->loggerFactory->get('comfyui')
        ->info('Workflow @id queued for execution with view mode @mode', [
          '@id' => $workflow_id,
          '@mode' => $view_mode,
        ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('comfyui')
        ->error('Failed to queue workflow: @error', [
          '@error' => $e->getMessage(),
        ]);
      return FALSE;
    }
  }
  
}
