<?php

namespace Drupal\comment_submissions_limit;

use Drupal\Core\Datetime\DrupalDateTime;

/**
 * Provides a value object for intervals (1 month, 14 days, etc).
 */
final class Interval {

  /**
   * The number.
   *
   * @var int
   */
  protected $number;

  /**
   * The unit.
   *
   * @var string
   */
  protected $unit;

  /**
   * Constructs a new Interval object.
   *
   * @param int $number
   *   The number.
   * @param string $unit
   *   The unit.
   */
  public function __construct(int $number, string $unit) {
    if (!in_array($unit, ['minute', 'hour', 'day', 'week', 'month', 'year'])) {
      throw new \InvalidArgumentException(sprintf('Invalid interval unit "%s" provided.', $unit));
    }

    $this->number = $number;
    $this->unit = $unit;
  }

  /**
   * Gets the number.
   *
   * @return int
   *   The number.
   */
  public function getNumber() : int {
    return $this->number;
  }

  /**
   * Gets the unit.
   *
   * @return string
   *   The unit.
   */
  public function getUnit() : string {
    return $this->unit;
  }

  /**
   * Gets the string representation of the interval.
   *
   * @return string
   *   The string representation of the interval.
   */
  public function __toString() : string {
    return $this->number . ' ' . $this->unit;
  }

  /**
   * Subtracts the interval from the given date.
   *
   * @param \Drupal\Core\Datetime\DrupalDateTime $date
   *   The date.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime
   *   The new date.
   */
  public function subtract(DrupalDateTime $date) : DrupalDateTime {
    /** @var \Drupal\Core\Datetime\DrupalDateTime $new_date */
    $new_date = clone $date;
    $new_date->modify('-' . $this->__toString());
    // Mar 31st - 1 month should Feb 28th, not Mar 3rd.
    if ($this->unit == 'month' && $new_date->format('d') !== $date->format('d')) {
      $new_date->modify('last day of previous month');
    }

    return $new_date;
  }

}
