<?php

namespace Drupal\comment_tracker;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\comment\Entity\Comment;
use Drupal\node\NodeInterface;

/**
 * Service to track comment read/unread stats per user.
 */
class CommentTrackerManager {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected AccountInterface $currentUser,
    protected CacheBackendInterface $cache
  ) {}

  /**
   * Returns comment stats (new/read/total) for an entity.
   *
   * "New" comments are those the user has not read yet.
   */
  public function getStats(EntityInterface $entity): object {
    $uid = (int) $this->currentUser->id();
    if ($uid === 0) {
      return (object) ['new' => 0, 'read' => 0, 'total' => 0];
    }

    $cid = "comment_tracker:{$entity->uuid()}:user:$uid";

    // Return cached stats.
    if ($cache = $this->cache->get($cid)) {
      return $cache->data;
    }

    $comment_storage = $this->entityTypeManager->getStorage('comment');
    $comment_type_storage = $this->entityTypeManager->getStorage('comment_type');
    $tracker_storage = $this->entityTypeManager->getStorage('comment_tracker');

    // Load all comments for this entity.
    $comments = $comment_storage->loadByProperties([
      'entity_id' => $entity->id(),
      'entity_type' => $entity->getEntityTypeId(),
    ]);

    $new = 0;
    $read = 0;


    foreach ($comments as $comment) {
      /** @var Comment $comment */

      if ($comment->getOwnerId() == $uid) {
        continue;
      }

      // Skip comment types not enabled for tracking.
      $comment_type = $comment_type_storage->load($comment->bundle());
      if (!$comment_type || !$comment_type->getThirdPartySetting('comment_tracker', 'enabled', FALSE)) {
        continue;
      }

      // Check if user already read this comment.
      $already_read = $tracker_storage->loadByProperties([
        'viewer_uid' => $uid,
        'comment_id' => $comment->id(),
      ]);


      if ($already_read) {
        $read++;
      } else {
        $new++;
      }
    }

    $result = (object) [
      'new' => $new,
      'read' => $read,
      'total' => $new + $read,
    ];

    // Cache per user + entity.
    $this->cache->set(
      $cid,
      $result,
      CacheBackendInterface::CACHE_PERMANENT,
      [
        "entity:{$entity->getEntityTypeId()}:{$entity->id()}",
        "user:$uid",
      ]
    );

    return $result;
  }

  /**
   * Marks a single comment as read for the current user.
   */
  public function markCommentRead(Comment $comment): void {
    $uid = (int) $this->currentUser->id();
    if ($uid === 0) {
      return;
    }

    \Drupal::logger('comment_tracker')->debug('Marking comment @cid as read for user @uid. Comment Author is @owner!', [
      '@owner' => $comment->getOwnerId(),
      '@uid' => $uid,
      '@cid' => $comment->id(),
    ]);
    
    if ($comment->getOwnerId() == $uid) {
      // Users should not mark their own comments as read.
      return;
    }

    $tracker_storage = $this->entityTypeManager->getStorage('comment_tracker');

    // Avoid duplicate records.
    $existing = $tracker_storage->loadByProperties([
      'viewer_uid' => $uid,
      'comment_id' => $comment->id(),
    ]);

    if (empty($existing)) {
      $record = $tracker_storage->create([
        'viewer_uid' => $uid,
        'comment_id' => $comment->id(),
        'viewed_at' => \Drupal::time()->getCurrentTime(),
      ]);
      $record->save();

      // Invalidate cache for this comment's entity.
      $entity = $comment->getCommentedEntity();
      \Drupal::service('cache_tags.invalidator')->invalidateTags([
        "entity:{$entity->getEntityTypeId()}:{$entity->id()}",
        "user:$uid",
      ]);


      \Drupal::service('cache_tags.invalidator')->invalidateTags([
        'comment_tracker:viewer:' . $uid,
        'comment:' . $comment->id(),
      ]);
      

    }
  }

  /**
   * Returns TRUE if comment tracking is enabled for at least one
   * comment field on the given node.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The node/entity to check.
   *
   * @return bool
   *   TRUE if comment tracker is enabled for its comment type, FALSE otherwise.
   */
  public function isTrackingEnabled(EntityInterface $entity): bool {
    if ($entity->getEntityTypeId() !== 'node') {
      return FALSE;
    }
  
    $comment_type_storage = $this->entityTypeManager->getStorage('comment_type');
    $field_definitions = $entity->getFieldDefinitions();
  
    foreach ($field_definitions as $field_name => $definition) {
  
      // Only process comment fields.
      if ($definition->getType() !== 'comment') {
        continue;
      }
  
      // Load the comment type configured on the field.
      $comment_type_id = $definition->getSetting('comment_type');
      if (!$comment_type_id) {
        continue;
      }
  
      /** @var \Drupal\comment\CommentTypeInterface $comment_type */
      $comment_type = $comment_type_storage->load($comment_type_id);
      if (!$comment_type) {
        continue;
      }
  
      // Check third-party setting "comment_tracker".
      $enabled = $comment_type->getThirdPartySetting('comment_tracker', 'enabled', FALSE);
  
      if ($enabled) {
        return TRUE; // At least one comment field is tracked → done.
      }
    }
  
    return FALSE;
  }

  /**
   * Returns TRUE if the given comment is new (unread) for the current user.
   */
  public function isNewComment(Comment $comment): bool {
    $uid = (int) $this->currentUser->id();
    if ($uid === 0) {
      return FALSE;
    }
  
    // Users do not track their own comments.
    if ($comment->getOwnerId() == $uid) {
      return FALSE;
    }
  
    $tracker_storage = $this->entityTypeManager->getStorage('comment_tracker');
  
    $existing = $tracker_storage->loadByProperties([
      'viewer_uid' => $uid,
      'comment_id' => (int) $comment->id(),
    ]);
 

    if (!empty($existing)) {

      $debug = reset($existing);

      $test = [
        'viewer_uid' => $debug->viewer_uid->target_id,
        'comment_id' => $debug->comment_id->target_id,
        'id' => $debug->id(),
      ];

      return FALSE;
    
    }

    return TRUE;

        
  }
  
}
