<?php

namespace Drupal\commerce_admin_app\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DrupalDateTime;

/**
 * Defines date range constants for analytics.
 */
class DateRange {
  const TODAY = 'today';
  const LAST_7_DAYS = 'last_7_days';
  const LAST_30_DAYS = 'last_30_days';
  const LAST_90_DAYS = 'last_90_days';
  const CUSTOM = 'custom';
}

/**
 * Service class for commerce sales analytics.
 */
class CommerceAnalyticsService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a new CommerceAnalyticsService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    Connection $database
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
  }

  /**
   * Gets date range based on the specified type.
   *
   * @param string $range_type
   *   Type of date range (use DateRange constants).
   * @param string|null $custom_start_date
   *   Custom start date in 'Y-m-d' format.
   * @param string|null $custom_end_date
   *   Custom end date in 'Y-m-d' format.
   *
   * @return array
   *   Associative array with 'start_date' and 'end_date'.
   */
  protected function getDateRange($range_type, $custom_start_date = NULL, $custom_end_date = NULL) {
    $end_date = date('Y-m-d');

    switch ($range_type) {
      case DateRange::TODAY:
        $start_date = $end_date;
        break;

      case DateRange::LAST_7_DAYS:
        $start_date = date('Y-m-d', strtotime('-7 days'));
        break;

      case DateRange::LAST_30_DAYS:
        $start_date = date('Y-m-d', strtotime('-30 days'));
        break;

      case DateRange::LAST_90_DAYS:
        $start_date = date('Y-m-d', strtotime('-90 days'));
        break;

      case DateRange::CUSTOM:
        if (!$custom_start_date || !$custom_end_date) {
          throw new \InvalidArgumentException('Custom date range requires both start and end dates.');
        }
        $start_date = $custom_start_date;
        $end_date = $custom_end_date;
        break;

      default:
        throw new \InvalidArgumentException('Invalid date range type specified.');
    }

    return [
      'start_date' => $start_date,
      'end_date' => $end_date,
    ];
  }

  /**
   * Gets total sales amount for a given date range.
   *
   * @param string $range_type
   *   Type of date range (use DateRange constants).
   * @param string|null $custom_start_date
   *   Custom start date in 'Y-m-d' format.
   * @param string|null $custom_end_date
   *   Custom end date in 'Y-m-d' format.
   *
   * @return float
   *   Total sales amount.
   */
  public function getTotalSales($range_type = DateRange::LAST_30_DAYS, $custom_start_date = NULL, $custom_end_date = NULL) {
    $date_range = $this->getDateRange($range_type, $custom_start_date, $custom_end_date);
    //print_r(strtotime($date_range['start_date'])); echo "====";
   // print_r(strtotime($date_range['end_date'] . ' 23:59:59'));
    //exit;
    $query = $this->database->select('commerce_order', 'co')
      ->condition('co.state', 'completed')
     
      ->condition('co.completed', strtotime($date_range['start_date']), '>=')
      ->condition('co.completed', strtotime($date_range['end_date'] . ' 23:59:59'), '<=');

      // Add expression to calculate total sales.
      $query->addExpression('SUM(co.total_price__number)', 'total_sales');

      $sql = $query->__toString();
      \Drupal::logger('commerce_admin_app')->info('Generated SQL: @query', ['@query' => $sql]);
      
      try {
          $result = $query->execute()->fetchField();
          return $result;
         // echo  $result;exit;
          //print_r($result);exit;
      } catch (\Exception $e) {
          \Drupal::logger('commerce_admin_app')->error('Query error: @message', ['@message' => $e->getMessage()]);
      }

  }

  /**
   * Gets average daily sales for a given date range.
   *
   * @param string $range_type
   *   Type of date range (use DateRange constants).
   * @param string|null $custom_start_date
   *   Custom start date in 'Y-m-d' format.
   * @param string|null $custom_end_date
   *   Custom end date in 'Y-m-d' format.
   *
   * @return float
   *   Average daily sales amount.
   */
  public function getAverageDailySales($range_type = DateRange::LAST_30_DAYS, $custom_start_date = NULL, $custom_end_date = NULL) {
    $date_range = $this->getDateRange($range_type, $custom_start_date, $custom_end_date);
    
    $total_sales = $this->getTotalSales($range_type, $custom_start_date, $custom_end_date);
    $days = (strtotime($date_range['end_date']) - strtotime($date_range['start_date'])) / (60 * 60 * 24) + 1;
    $avg_sales = $days > 0 ? $total_sales / $days : 0;
    return floor($avg_sales);
  }

  /**
   * Gets trending items based on sales quantity for a given date range.
   *
   * @param string $range_type
   *   Type of date range (use DateRange constants).
   * @param string|null $custom_start_date
   *   Custom start date in 'Y-m-d' format.
   * @param string|null $custom_end_date
   *   Custom end date in 'Y-m-d' format.
   * @param int $limit
   *   Number of items to return.
   *
   * @return array
   *   Array of trending products with their details.
   */
  public function getTrendingItems($range_type = DateRange::LAST_30_DAYS, $custom_start_date = NULL, $custom_end_date = NULL, $limit = 5) {
    $date_range = $this->getDateRange($range_type, $custom_start_date, $custom_end_date);

    // Get order item IDs with quantities.
    $query = $this->database->select('commerce_order_item', 'coi');
    $query->join('commerce_order', 'co', 'co.order_id = coi.order_id');
    $query->fields('coi', ['purchased_entity'])
      ->condition('co.state', 'completed')
      ->condition('co.completed', strtotime($date_range['start_date']), '>=')
      ->condition('co.completed', strtotime($date_range['end_date']. ' 23:59:59'), '<=')
      ->groupBy('coi.purchased_entity')
      ->orderBy('quantity_sold', 'DESC')
      ->range(0, $limit);
    $query->addExpression('SUM(coi.quantity)', 'quantity_sold');

    $results = $query->execute()->fetchAll();
   //print_r($results);exit;
    $trending_items = [];
    foreach ($results as $result) {
      $product_variation = $this->entityTypeManager
        ->getStorage('commerce_product_variation')
        ->load($result->purchased_entity);
       //print_r( $product_variation);exit;
      if (!$product_variation) {
        continue;
      }
      //print_r($product_variation->get('field_images')->entity->getFileUri());exit;
      $product = $product_variation->getProduct();
      //print_r($product);exit;
      // Get product image.
      $image_url = '';
   
      if ($product_variation->hasField('field_images') && !$product_variation->get('field_images')->isEmpty()) {
        $file = $product_variation->get('field_images')->entity;
        if ($file instanceof \Drupal\file\FileInterface) {
          $uri = $file->getFileUri();
          $image_url = file_create_url($uri);
        }
      }

      $trending_items[] = [
        'title' => $product->getTitle(),
        'image_url' => $image_url,
        'price' => $product_variation->getPrice()->getNumber(),
        'currency_code' => $product_variation->getPrice()->getCurrencyCode(),
        'quantity_sold' => $result->quantity_sold,
      ];
    }
      //print_r($trending_items);exit;
    return $trending_items;
  }
  /**
   * Gets order count chart data for different time ranges.
   *
   * @param string $range_type
   *   Type of date range (use DateRange constants).
   * @param string|null $custom_start_date
   *   Custom start date in 'Y-m-d' format.
   * @param string|null $custom_end_date
   *   Custom end date in 'Y-m-d' format.
   *
   * @return array
   *   Array of chart data with labels and order counts.
   */
  public function getOrderChartData($range_type = DateRange::LAST_30_DAYS, $custom_start_date = NULL, $custom_end_date = NULL) {
    $date_range = $this->getDateRange($range_type, $custom_start_date, $custom_end_date);
    $chart_data = [];

    switch ($range_type) {
      case DateRange::TODAY:
        // Split day into 6 intervals (4-hour blocks)
        $intervals = [
          '00:00' => '12:00 AM',
          '04:00' => '4:00 AM',
          '08:00' => '8:00 AM',
          '12:00' => '12:00 PM',
          '16:00' => '4:00 PM',
          '20:00' => '8:00 PM',
        ];

        foreach ($intervals as $time => $label) {
          $start_time = strtotime(date('Y-m-d') . ' ' . $time);
          $end_time = $start_time + (4 * 60 * 60) - 1;

          $count = $this->getOrderCountForTimeRange($start_time, $end_time);
          $chart_data[] = [
            'label' => $label,
            'value' => $count,
          ];
        }
        break;

      case DateRange::LAST_7_DAYS:
        // Daily data for last 7 days
        for ($i = 6; $i >= 0; $i--) {
          $date = strtotime("-$i days");
          $start_time = strtotime(date('Y-m-d 00:00:00', $date));
          $end_time = strtotime(date('Y-m-d 23:59:59', $date));

          $count = $this->getOrderCountForTimeRange($start_time, $end_time);
          $chart_data[] = [
            'label' => date('D', $date),
            'value' => $count,
          ];
        }
        break;

      case DateRange::LAST_30_DAYS:
        // Weekly data for last 30 days
        for ($i = 0; $i < 4; $i++) {
          $week_start = strtotime("-" . (($i + 1) * 7) . " days");
          $week_end = strtotime("-" . ($i * 7) . " days");

          $count = $this->getOrderCountForTimeRange($week_start, $week_end);
          $chart_data[] = [
            'label' => 'Week ' . (4 - $i),
            'value' => $count,
          ];
        }
        break;

      case DateRange::LAST_90_DAYS:
        // Monthly data for last 90 days
        for ($i = 2; $i >= 0; $i--) {
          $month_start = strtotime(date('Y-m-01', strtotime("-$i months")));
          $month_end = strtotime(date('Y-m-t', strtotime("-$i months")));

          $count = $this->getOrderCountForTimeRange($month_start, $month_end);
          $chart_data[] = [
            'label' => date('M', $month_start),
            'value' => $count,
          ];
        }
        break;

      case DateRange::CUSTOM:
        if (!$custom_start_date || !$custom_end_date) {
          return [];
        }

        $start = strtotime($custom_start_date);
        $end = strtotime($custom_end_date);
        $days_diff = floor(($end - $start) / (60 * 60 * 24));

        // Generate daily data points for the custom range
        for ($i = 0; $i <= $days_diff; $i++) {
          $current_date = strtotime("+$i days", $start);
          $day_start = strtotime(date('Y-m-d 00:00:00', $current_date));
          $day_end = strtotime(date('Y-m-d 23:59:59', $current_date));

          $count = $this->getOrderCountForTimeRange($day_start, $day_end);
          $chart_data[] = [
            'label' => date('M d', $current_date),
            'value' => $count,
          ];
        }
        break;
    }

    return $chart_data;
  }

  /**
   * Helper function to get order count for a specific time range.
   *
   * @param int $start_time
   *   Start timestamp.
   * @param int $end_time
   *   End timestamp.
   *
   * @return int
   *   Number of orders in the time range.
   */
  protected function getOrderCountForTimeRange($start_time, $end_time) {
    return $this->database->select('commerce_order', 'co')
      ->condition('co.state', 'completed')
      ->condition('co.completed', $start_time, '>=')
      ->condition('co.completed', $end_time, '<=')
      ->countQuery()
      ->execute()
      ->fetchField();
  }
}

//=====usage=======//
//======usage=====