<?php

namespace Drupal\commerce_admin_app\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\commerce_order\Entity\Order;

/**
 * Service class for fetching Commerce order data formatted for Flutter UI.
 */
class CommerceOrderService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a new CommerceOrderService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Get order data for Flutter UI.
   *
   * @param string $timeRange
   *   Time range: 'Today', 'Weekly', or 'Monthly'.
   *
   * @return array
   *   Array containing chart data and order list formatted for Flutter UI.
   */
  public function getOrderData($timeRange = 'Weekly') {
    // Get raw order data
    $orders = $this->fetchOrders($timeRange);
    //echo "time-range--".$timeRange;
     //echo sizeof($orders);exit;
     //print_r($orders );
    // Format data for Flutter UI
    return [
      'chartData' => $this->formatChartData($orders, $timeRange),
      'orderList' => $this->formatOrderList($orders),
    ];
  }

  /**
   * Fetch orders based on time range.
   */
  private function fetchOrders($timeRange) {
    $storage = $this->entityTypeManager->getStorage('commerce_order');
    $query = $storage->getQuery();

    // Set time range conditions
    $end = new DrupalDateTime('tomorrow');
    switch ($timeRange) {
      case 'Today':
        $start = new DrupalDateTime('today');
        break;
      case 'Weekly':
        $start = new DrupalDateTime('-7 days');
        break;
      case 'Monthly':
        $start = new DrupalDateTime('-6 months');
        break;
      default:
        $start = new DrupalDateTime('-7 days');
    }

    $query->condition('state', 'completed')
      ->condition('placed', $start->getTimestamp(), '>=')
      ->condition('placed', $end->getTimestamp(), '<')
      ->sort('placed', 'DESC');

    $order_ids = $query->execute();
    return $storage->loadMultiple($order_ids);
  }

  /**
   * Format chart data for Flutter UI.
   */
  private function formatChartData(array $orders, $timeRange) {
    $data = [];
    //echo "sizeo of-tm--".$timeRange;
    switch ($timeRange) {
      case 'Today':
        // Initialize hourly buckets
        for ($i = 0; $i < 24; $i++) {
          $hour = sprintf('%02d:00', $i);
          $data[$hour] = 0;
        }
        
        // Process orders
        foreach ($orders as $order) {
          $hour = date('H:00', $order->getPlacedTime());
          if (isset($data[$hour])) {
            $data[$hour] += $order->getTotalPrice()->getNumber();
          }
        }
        
        // Format for Flutter
        return array_map(function($hour, $value) {
          return [
            'date' => $hour,
            'value' => round($value, 2),
          ];
        }, array_keys($data), array_values($data));
        
      case 'Weekly':
        // Initialize daily buckets for last 7 days
        for ($i = 6; $i >= 0; $i--) {
          $date = date('M d', strtotime("-$i days"));
          $data[$date] = 0;
        }
        
        // Process orders
        foreach ($orders as $order) {
          $date = date('M d', $order->getPlacedTime());
          if (isset($data[$date])) {
            $data[$date] += $order->getTotalPrice()->getNumber();
          }
        }
        break;
        
      case 'Monthly':
        // Initialize monthly buckets
        for ($i = 5; $i >= 0; $i--) {
          $date = date('M', strtotime("-$i months"));
          $data[$date] = 0;
        }
        
        // Process orders
        foreach ($orders as $order) {
          $date = date('M', $order->getPlacedTime());
          if (isset($data[$date])) {
            $data[$date] += $order->getTotalPrice()->getNumber();
          }
        }
        break;
    }

    //echo "sizeo of---".sizeof($data);
    // Format data for Flutter chart
    $chartData = [];
    foreach ($data as $date => $value) {
      $chartData[] = [
        'date' => $date,
        'value' => round($value, 2),
      ];
      //echo "date-".$date."-range-".$timeRange;
    }

    return $chartData;
  }

  /**
   * Format order list for Flutter UI.
   */
  private function formatOrderList(array $orders) {
    $orderList = [];

    foreach ($orders as $order) {
      $customer = $order->getCustomer();
      $name = $customer ? $customer->getDisplayName() : 'Guest';
      
      // Format date based on order age
      $placedTime = $order->getPlacedTime();
      if (date('Y-m-d') === date('Y-m-d', $placedTime)) {
        // Today - show time
        $formattedDate = date('g:i A', $placedTime);
      } 
      elseif (date('Y') === date('Y', $placedTime)) {
        // This year - show month and day
        $formattedDate = date('M d', $placedTime);
      } 
      else {
        // Different year - show with year
        $formattedDate = date('M d, Y', $placedTime);
      }

      $orderList[] = [
        'initials' => $this->getInitials($name),
        'name' => $name,
        'date' => $formattedDate,
        'amount' => round($order->getTotalPrice()->getNumber(), 2),
      ];
    }

    return $orderList;
  }

  /**
   * Helper function to get initials from a name.
   */
  private function getInitials($name) {
    $words = explode(' ', trim($name));
    $initials = '';
    foreach ($words as $word) {
      $initials .= strtoupper(substr($word, 0, 1));
    }
    return substr($initials, 0, 2);
  }
}