<?php

declare(strict_types=1);

namespace Drupal\commerce_advancedqueue\EventSubscriber;

use Drupal\advancedqueue\Event\AdvancedQueueEvents;
use Drupal\advancedqueue\Event\JobEvent;
use Drupal\advancedqueue\Job;
use Drupal\commerce\InboxMessage;
use Drupal\commerce\InboxMessageStorageInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Allows pushing queue job failures into commerce inbox.
 */
final class JobNotificationSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Constructs a JobNotificationSubscriber object.
   */
  public function __construct(
    private readonly InboxMessageStorageInterface $inboxMessageStorage,
    private readonly LoggerInterface $logger,
  ) {}

  public function notifyOnFailure(JobEvent $event): void {
    $job = $event->getJob();
    // @todo: Skip jobs types that are not commerce related.
    if ($job->getState() !== Job::STATE_FAILURE) {
      return;
    }
    try {
      $message = [];
      $message['id'] = sprintf('advancedqueue-%s', $job->getQueueId());
      $message['subject'] = $this->t('Background job failed');
      $message['message'] = $this->t('Job in the queue @queue failed with the message @message.',
        [
          '@queue' => $job->getQueueId(),
          '@message' => $job->getMessage(),
        ]);
      $message['cta_text'] = $this->t('See all failed jobs in @queue', [
        '@queue' => $job->getQueueId(),
      ]);
      $message['cta_link'] = Url::fromRoute('view.advancedqueue_jobs.page_1',
        [
          'arg_0' => $job->getQueueId(),
        ],
        [
          'query' => [
            'state' => Job::STATE_FAILURE,
          ],
        ]
      )->toString();
      $message['send_date'] = \Drupal::time()->getCurrentTime();
      $message['state'] = 'unread';
      $inbox_message = InboxMessage::fromArray($message);
      $this->inboxMessageStorage->save($inbox_message);
    }
    catch (\Throwable $exception) {
      $this->logger->error('Unable to create inbox message: @message', ['@message' => $exception->getMessage()]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      // @todo: Once https://drupal.org/i/3255444 is in, change the job type.
      AdvancedQueueEvents::POST_PROCESS => ['notifyOnFailure'],
    ];
  }

}
