# Commerce AI Suite

**Parent module for AI-powered commerce features**

The Commerce AI Suite module provides a foundation and shared services for
integrating artificial intelligence capabilities into Drupal Commerce. It
includes common services for Google Cloud Platform integrations and serves as
the parent for specialized AI feature submodules.

## Introduction

Commerce AI Suite is designed as a modular system where each AI feature is
implemented as a submodule. This architecture allows you to:

- Enable only the AI features you need
- Share common services (GCS client) across features
- Maintain a consistent configuration approach
- Scale with additional AI features over time

## Shared Services

### GCS Client Service (`commerce_ai_suite.gcs_client`)

Provides centralized Google Cloud Storage functionality for all submodules:

- Upload files with retry logic and exponential backoff
- List and delete GCS objects
- Configurable per-submodule settings
- Secure service account authentication via Key module

### Key Helper Service (`commerce_ai_suite.key_helper`)

Provides centralized key management for service account credentials:

- Loads service account keys from Drupal Key module
- Validates service account JSON structure
- Consistent error handling across submodules

**Usage in submodules:**
```php
$gcs_client = \Drupal::service('commerce_ai_suite.gcs_client');
$result = $gcs_client->uploadFileWithRetry(
  'my_submodule.settings',  // Config object name
  'my-bucket-name',         // Bucket name
  $file_content,            // Content to upload
  'path/to/file.jsonl'      // Destination path
);
```

## Current Submodules

### Commerce AI Suite Product Recommender

Provides personalized product recommendations using AI providers through a
flexible plugin architecture. Google Vertex AI is included by default, and
developers can add custom providers (OpenAI ChatGPT, Ollama, etc.).

**Features:**
- **Pluggable AI providers:** Support for multiple AI services via plugin system
- Analyzes user order history for personalized recommendations
- Optional product catalog export to GCS (for compatible providers)
- Returns calculated prices with all Commerce adjustments
- Multi-currency support with currency codes
- Complete product data (SKU, title, URL, image, price) in single API call
- AJAX-enabled configuration form with conditional field display
- Block for displaying personalized suggestions
- REST API endpoint for external/headless integrations
- Drush commands and optional cron integration
- Extensible: Developers can add custom AI providers (see Developer Guide)

[View Documentation](modules/commerce_ai_suite_product_recommender/README.md)

[Developer Guide for Custom Providers](
  modules/commerce_ai_suite_product_recommender/DEVELOPER_GUIDE.md)

## Future Submodules (Roadmap)

### Commerce AI Suite Chatbot (Planned)
AI-powered customer support chatbot with:
- Product inquiry handling
- Order status lookups
- Support ticket creation
- Multi-language support

### Commerce AI Suite Search (Planned)
Intelligent product search with:
- Natural language queries
- Semantic search capabilities
- Personalized result ranking
- Voice search support

### Commerce AI Suite Content (Planned)
Automated content generation for:
- Product descriptions
- SEO meta tags
- Category descriptions
- Marketing copy

## Installation

### Requirements

- Drupal 10 or 11
- Commerce 2.x
- Key module 1.x
- PHP 8.1 or higher
- Google Cloud Platform account (for GCS features)

### Install via Composer

```bash
cd /path/to/drupal
composer require google/cloud-storage:^1.30
```

### Enable the Parent Module

```bash
drush en commerce_ai_suite -y
```

Then enable the specific submodule(s) you need:

```bash
drush en commerce_ai_suite_product_recommender -y
```

## Configuration

Each submodule has its own configuration form. The parent module provides:

- Common service definitions
- Shared GCS configuration patterns
- Logging infrastructure

Navigate to submodule-specific settings:
- Product Recommender: `/admin/commerce/config/ai-product-recommender`

## Architecture

### Directory Structure

```
commerce_ai_suite/
├── composer.json              # Parent module dependencies
├── commerce_ai_suite.info.yml       # Module definition
├── commerce_ai_suite.module         # Hook implementations
├── commerce_ai_suite.services.yml   # Shared services
├── README.md                  # This file
├── src/
│   └── Service/
│       ├── GcsClientService.php     # Shared GCS client
│       └── KeyHelper.php            # Shared key management
└── modules/                   # Submodules directory
    ├── commerce_ai_suite_product_recommender/
    │   ├── composer.json
    │   ├── commerce_ai_suite_product_recommender.info.yml
    │   ├── README.md
    │   ├── INSTALL.md
    │   └── src/
    │       ├── Service/
    │       ├── Plugin/
    │       ├── Form/
    │       └── Commands/
    └── [future submodules]/
```

### Service Architecture

```
Parent Module (commerce_ai_suite)
├── commerce_ai_suite.gcs_client (shared)
├── commerce_ai_suite.key_helper (shared)
│
└── Submodules depend on parent services
    ├── commerce_ai_suite_product_recommender.ai_provider_manager
    │   └── Manages AI provider plugins
    ├── commerce_ai_suite_product_recommender.product_exporter
    │   └── Uses @commerce_ai_suite.gcs_client
    ├── commerce_ai_suite_product_recommender.recommendation_service
    │   └── Uses AI provider plugins
    └── AI Provider Plugins (VertexAi, etc.)
        └── Use @commerce_ai_suite.key_helper
```

### Dependency Flow

```
commerce_ai_suite (parent)
  └── Provides: GCS client, Key helper
      │
      └── commerce_ai_suite_product_recommender (submodule)
          └── Requires: commerce_ai_suite
          └── Uses: commerce_ai_suite.gcs_client, commerce_ai_suite.key_helper
```

## Extending Commerce AI Suite

### Creating a New Submodule

1. **Create the submodule directory:**
   ```bash
   mkdir -p modules/commerce_ai_suite_my_feature/src/Service
   ```

2. **Create `commerce_ai_suite_my_feature.info.yml`:**
   ```yaml
   name: Commerce AI Suite My Feature
   type: module
   description: 'My custom AI feature.'
   core_version_requirement: ^10 || ^11
   package: Commerce AI Suite
   dependencies:
     - commerce_ai_suite:commerce_ai_suite
   ```

3. **Use shared services in your code:**
   ```php
   namespace Drupal\commerce_ai_suite_my_feature\Service;

   use Drupal\commerce_ai_suite\Service\GcsClientService;

   class MyService {
     protected $gcsClient;

     public function __construct(GcsClientService $gcs_client) {
       $this->gcsClient = $gcs_client;
     }

     public function doSomething() {
       $this->gcsClient->uploadFile(
         'commerce_ai_suite_my_feature.settings',
         'my-bucket',
         $content,
         'path/file.txt'
       );
     }
   }
   ```

4. **Define your services in `commerce_ai_suite_my_feature.services.yml`:**
   ```yaml
   services:
     commerce_ai_suite_my_feature.my_service:
       class: Drupal\commerce_ai_suite_my_feature\Service\MyService
       arguments: ['@commerce_ai_suite.gcs_client']
   ```

## Best Practices

### For Submodule Development

1. **Depend on parent module:** Always add `commerce_ai_suite:commerce_ai_suite` to
   dependencies
2. **Use shared services:** Leverage `commerce_ai_suite.gcs_client` instead of
   creating your own
3. **Follow naming conventions:** Prefix services with `commerce_ai_suite_[feature]`
4. **Package correctly:** Set `package: Commerce AI Suite` in info.yml
5. **Document thoroughly:** Provide README and INSTALL guides

### For Configuration

1. **Use Key module:** For sensitive credentials (API keys, service accounts)
2. **Configuration schema:** Always define schema for settings
3. **Validation:** Implement form validation for JSON, URLs, etc.
4. **Defaults:** Provide sensible default values

### For Services

1. **Dependency Injection:** Never use static service calls
2. **Type hints:** Always type-hint injected services
3. **Logging:** Use injected logger channels
4. **Error handling:** Catch exceptions, log errors, return gracefully

## Troubleshooting

### "Service 'commerce_ai_suite.gcs_client' not found"

**Solution:** Ensure the parent `commerce_ai_suite` module is enabled:
```bash
drush en commerce_ai_suite -y
drush cr
```

### Module appears in wrong package

**Solution:** Verify `package: Commerce AI Suite` in your `.info.yml` file

### GCS operations failing

**Solution:** Check service account credentials:
1. Verify JSON key is valid
2. Confirm service account has required permissions
3. Check bucket name configuration
4. Review logs: `drush watchdog:show --type=commerce_ai_suite`

## Contributing

When contributing new AI features:

1. Create a new submodule in `modules/`
2. Follow the directory structure of existing submodules
3. Depend on `commerce_ai_suite` parent module
4. Use shared services where applicable
5. Document your feature thoroughly
6. Add tests for your functionality

## Security Considerations

- **Service Account Keys:** Never commit to version control
- **API Keys:** Use Key module for sensitive credentials
- **Data Privacy:** Anonymize user data before sending to external AI services
- **Access Control:** Implement proper permission checks
- **Rate Limiting:** Consider implementing for external API calls

## Performance

- **Caching:** Implement per-user caching for AI responses
- **Queue API:** Use for long-running operations
- **Chunking:** Break large operations into manageable pieces
- **Retry Logic:** Implement exponential backoff for API calls

## Credits

- **Developed by:** Acro Commerce
- **Client:** Concept2
- **Technology Partners:** Google Cloud Platform, Vertex AI

## License

GPL-2.0-or-later

## Support

For issues, feature requests, or questions about Commerce AI Suite, contact the
Acro Commerce development team.
