# Developer Guide: Creating Custom AI Provider Plugins

This guide explains how to extend the Commerce AI Suite Product Recommender
module to support additional AI providers beyond the default Google
Vertex AI.

## Overview

The Commerce AI Suite Product Recommender uses Drupal's plugin system to
support multiple AI providers. This allows developers to add support
for different AI services (e.g., OpenAI ChatGPT, Anthropic Claude,
Ollama, etc.) without modifying the core module code.

## Plugin Architecture

### Key Components

1. **AiProviderInterface** - Defines the contract all providers must
   implement
2. **AiProviderBase** - Abstract base class with common functionality
3. **AiProviderManager** - Plugin manager that discovers and instantiates
   providers
4. **@AiProvider Annotation** - Declares plugin metadata

## Creating a New AI Provider Plugin

### Step 1: Create the Plugin Class

Create a new PHP class in your custom module at:
```
src/Plugin/AiProvider/YourProvider.php
```

### Step 2: Implement the Plugin

Here's a complete example for a ChatGPT provider:

```php
// phpcs:disable
<?php

namespace Drupal\your_module\Plugin\AiProvider;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderBase;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * OpenAI ChatGPT provider plugin.
 *
 * @AiProvider(
 *   id = "chatgpt",
 *   label = @Translation("OpenAI ChatGPT"),
 *   description = @Translation("Uses OpenAI ChatGPT API for product
 *     recommendations."),
 *   weight = 10
 * )
 */
class ChatGpt extends AiProviderBase {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    ClientInterface $http_client,
  ) {
    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $config_factory,
      $logger_factory
    );
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('http_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getRecommendations($user_history) {
    $config = $this->getConfig();
    $api_key = $config->get('chatgpt_api_key');

    if (empty($api_key)) {
      $this->logger->error('ChatGPT API key is not configured.');
      return NULL;
    }

    try {
      // Prepare the prompt with user history.
      $prompt = $this->preparePrompt($user_history);

      // Make API request to OpenAI.
      $response = $this->httpClient->post(
        'https://api.openai.com/v1/chat/completions',
        [
          'headers' => [
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type' => 'application/json',
          ],
          'json' => [
            'model' => 'gpt-4',
            'messages' => [
              [
                'role' => 'system',
                'content' => 'You are a product recommendation ' .
                'assistant. Return recommendations as JSON.',
              ],
              [
                'role' => 'user',
                'content' => $prompt,
              ],
            ],
            'temperature' => 0.7,
          ],
          'timeout' => 30,
        ]);

      return $this->processResponse($response);
    }
    catch (\Exception $e) {
      $this->logger->error(
        'ChatGPT API error: @msg',
        ['@msg' => $e->getMessage()]
      );
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getConfigurationForm() {
    $config = $this->getConfig();

    return [
      'chatgpt' => [
        '#type' => 'fieldset',
        '#title' => $this->t('ChatGPT Settings'),
        '#tree' => FALSE,
        'chatgpt_api_key' => [
          '#type' => 'textfield',
          '#title' => $this->t('ChatGPT API Key'),
          '#default_value' => $config->get('chatgpt_api_key'),
          '#required' => TRUE,
          '#description' => $this->t(
            'Your OpenAI API key. Get one at
            <a href="@url" target="_blank">OpenAI Platform</a>.',
            ['@url' => 'https://platform.openai.com/api-keys']
          ),
        ],
        'chatgpt_model' => [
          '#type' => 'select',
          '#title' => $this->t('Model'),
          '#options' => [
            'gpt-4' => 'GPT-4',
            'gpt-4-turbo' => 'GPT-4 Turbo',
            'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
          ],
          '#default_value' => $config->get('chatgpt_model') ?: 'gpt-4',
          '#required' => TRUE,
        ],
        'chatgpt_prompt' => [
          '#type' => 'textarea',
          '#title' => $this->t('System Prompt'),
          '#default_value' => $config->get('chatgpt_prompt'),
          '#required' => TRUE,
          '#description' => $this->t(
            'The base prompt template. Use [HISTORY_JSON] as a
            placeholder for user history.'
          ),
          '#rows' => 5,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfiguration(array $config) {
    $errors = [];

    if (empty($config['chatgpt_api_key'])) {
      $errors['chatgpt_api_key'] = $this->t('ChatGPT API Key is required.');
    }

    if (empty($config['chatgpt_prompt'])) {
      $errors['chatgpt_prompt'] = $this->t('System Prompt is required.');
    }

    return $errors;
  }

  /**
   * {@inheritdoc}
   */
  public function isConfigured() {
    $config = $this->getConfig();
    return !empty($config->get('chatgpt_api_key')) &&
           !empty($config->get('chatgpt_prompt'));
  }

  /**
   * Prepares the prompt with user history.
   *
   * @param string $user_history
   *   JSON-encoded user history.
   *
   * @return string
   *   The prepared prompt.
   */
  protected function preparePrompt($user_history) {
    $config = $this->getConfig();
    $prompt_template = $config->get('chatgpt_prompt');
    return str_replace('[HISTORY_JSON]', $user_history, $prompt_template);
  }

  /**
   * Processes the API response.
   *
   * @param \Psr\Http\Message\ResponseInterface $response
   *   The HTTP response.
   *
   * @return array|null
   *   Array of recommendations or NULL on failure.
   */
  protected function processResponse($response) {
    if ($response->getStatusCode() !== 200) {
      $this->logger->error('ChatGPT API returned status @code', [
        '@code' => $response->getStatusCode(),
      ]);
      return NULL;
    }

    $body = json_decode($response->getBody()->getContents(), TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      $this->logger->error('Failed to parse ChatGPT response');
      return NULL;
    }

    // Extract recommendations from ChatGPT response.
    $content = $body['choices'][0]['message']['content'] ?? '';
    $recommendations = json_decode($content, TRUE);

    if (
      json_last_error() !== JSON_ERROR_NONE ||
      !isset($recommendations['recommendations'])
    ) {
      $this->logger->warning(
        'Invalid recommendation format from ChatGPT'
      );
      return [];
    }

    return $recommendations['recommendations'];
  }

}

// phpcs:enable
```

### Step 3: Update Configuration Schema

Add your provider's configuration fields to the schema at:
```
config / schema / your_module . schema . yml
```

```yaml
commerce_ai_suite_product_recommender . settings:
  type: config_object
  label: 'Commerce AI Suite Product Recommender settings'
  mapping:
    chatgpt_api_key:
      type: string
      label: 'ChatGPT API Key'
    chatgpt_model:
      type: string
      label: 'ChatGPT Model'
    chatgpt_prompt:
      type: text
      label: 'ChatGPT System Prompt'
```

### Step 4: Update Settings Form Submit Handler

The settings form in `commerce_ai_suite_product_recommender` needs to save
your provider's configuration. Update the `submitForm()` method in
`SettingsForm . php` to include your fields:

```php
// phpcs:disable
public function submitForm(array &$form, FormStateInterface $form_state) {
  $config = $this->config('commerce_ai_suite_product_recommender.settings');
  $config
    ->set('ai_provider', $form_state->getValue('ai_provider'))
    // ... existing fields ...
    ->set('chatgpt_api_key', $form_state->getValue('chatgpt_api_key'))
    ->set('chatgpt_model', $form_state->getValue('chatgpt_model'))
    ->set('chatgpt_prompt', $form_state->getValue('chatgpt_prompt'))
    ->save();

  parent::submitForm($form, $form_state);
}
// phpcs:enable
```

## Plugin Annotation Reference

```php
// phpcs:disable
/**
 * @AiProvider(
 *   id = "unique_provider_id",
 *   label = @Translation("Human Readable Name"),
 *   description = @Translation("Brief description of the provider"),
 *   weight = 10
 * )
 */
// phpcs:enable
```

- **id**: Unique machine name for the provider
- **label**: Human-readable name shown in the UI
- **description**: Brief description shown in the settings form
- **weight**: Controls the order in dropdowns (lower = higher priority)

## Required Methods

### getRecommendations($user_history)

This is the core method that communicates with your AI service.

**Parameters:**
- `$user_history`: JSON-encoded string containing user's order history

**Returns:**
- Array of recommendation objects, each with:
  - `id`: Product ID
  - `reason`: Why this product is recommended
- NULL on error

**Example recommendation format:**
```php
// phpcs:disable
[
  ['id' => 123, 'reason' => 'Complements recent purchase'],
  ['id' => 456, 'reason' => 'Popular in your category'],
]
// phpcs:enable
```

### getConfigurationForm()

Returns a Drupal Form API array with provider-specific settings.

**Returns:**
- Form API array with configuration fields
- Fields will be automatically added to the settings form
- Use `$this->getConfig()` to load current values

### validateConfiguration(array $config)

Validates the provider's configuration.

**Parameters:**
- `$config`: Array of form values to validate

**Returns:**
- Array of error messages keyed by field name
- Empty array if validation passes

### isConfigured()

Checks if the provider has all required configuration.

**Returns:**
- TRUE if the provider is ready to use
- FALSE if configuration is missing or invalid

## Helper Methods from AiProviderBase

The base class provides these helper methods:

### getConfig()

Returns the module's configuration object.

```php
// phpcs:disable
$config = $this->getConfig();
$api_key = $config->get('your_api_key_field');
// phpcs:enable
```

### getLabel()

Returns the plugin's label from the annotation.

### getDescription()

Returns the plugin's description from the annotation.

## Best Practices

### 1. Error Handling

Always wrap API calls in try-catch blocks and log errors:

```php
// phpcs:disable
try {
  $response = $this->httpClient->post($url, $options);
  return $this->processResponse($response);
}
catch (\Exception $e) {
  $this->logger->error('API error: @msg', ['@msg' => $e->getMessage()]);
  return NULL;
}
// phpcs:enable
```

### 2. Configuration Validation

Implement thorough validation in `validateConfiguration()`:

```php
// phpcs:disable
public function validateConfiguration(array $config) {
  $errors = [];

  if (empty($config['api_key'])) {
    $errors['api_key'] = $this->t('API Key is required.');
  }

  // Validate format if needed.
  if (
    !empty($config['api_key']) &&
    !preg_match('/^sk-[a-zA-Z0-9]+$/', $config['api_key'])
  ) {
    $errors['api_key'] = $this->t('API Key format is invalid.');
  }

  return $errors;
}
// phpcs:enable
```

### 3. Response Parsing

Be defensive when parsing API responses:

```php
// phpcs:disable
protected function processResponse($response) {
  if ($response->getStatusCode() !== 200) {
    $this->logger->error(
      'Unexpected status: @code',
      ['@code' => $response->getStatusCode()]
    );
    return NULL;
  }

  $body = json_decode($response->getBody()->getContents(), TRUE);

  if (json_last_error() !== JSON_ERROR_NONE) {
    $this->logger->error(
      'JSON parse error: @error',
      ['@error' => json_last_error_msg()]
    );
    return NULL;
  }

  // Validate expected structure.
  if (
    !isset($body['recommendations']) ||
    !is_array($body['recommendations'])
  ) {
    $this->logger->warning('Invalid response structure');
    return [];
  }

  return $body['recommendations'];
}
// phpcs:enable
```

### 4. Timeout Configuration

Always set reasonable timeouts for API calls:

```php
// phpcs:disable
$response = $this->httpClient->post($url, [
  'timeout' => 30,  // 30 seconds
  'json' => $request_data,
]);
// phpcs:enable
```

### 5. Sensitive Data

Never expose API keys or credentials in logs:

```php
// phpcs:disable
// Bad - exposes key
$this->logger->info('Using key: @key', ['@key' => $api_key]);

// Good - masks sensitive data
$this->logger->info(
  'API key configured: @status',
  ['@status' => empty($api_key) ? 'no' : 'yes']
);
// phpcs:enable
```

## Testing Your Provider

### 1. Enable the Module

After creating your plugin, clear Drupal's cache:

```bash
drush cr
```

### 2. Check Plugin Discovery

Verify your plugin is discovered:

```php
// phpcs:disable
$provider_manager = \Drupal::service(
  'commerce_ai_suite_product_recommender.ai_provider_manager'
);
$providers = $provider_manager->getAvailableProviders();
// Should include your provider
// phpcs:enable
```

### 3. Test Configuration Form

1. Navigate to ` / admin / commerce / config / commerce - ai - product - recommender`
2. Select your provider from the dropdown
3. Verify your configuration fields appear
4. Save and test validation

### 4. Test Recommendations

Use Drush or a custom test script:

```php
// phpcs:disable
$recommendation_service = \Drupal::service(
  'commerce_ai_suite_product_recommender.recommendation_service'
);
$recommendations = $recommendation_service
  ->getRecommendationsForCurrentUser();
// phpcs:enable
```

## Example: Ollama Local AI Provider

Here's a simpler example using a local Ollama instance:

```php
// phpcs:disable
/**
 * Ollama local AI provider plugin.
 *
 * @AiProvider(
 *   id = "ollama",
 *   label = @Translation("Ollama (Local)"),
 *   description = @Translation("Uses local Ollama instance for
 *     recommendations."),
 *   weight = 20
 * )
 */
class Ollama extends AiProviderBase {

  public function getRecommendations($user_history) {
    $config = $this->getConfig();
    $endpoint = $config->get('ollama_endpoint') ?: 'http://localhost:11434';

    try {
      $response = $this->httpClient->post("$endpoint/api/generate", [
        'json' => [
          'model' => $config->get('ollama_model') ?: 'llama2',
          'prompt' => $this->preparePrompt($user_history),
          'stream' => FALSE,
        ],
        'timeout' => 60,
      ]);

      return $this->processResponse($response);
    }
    catch (\Exception $e) {
      $this->logger->error('Ollama error: @msg', ['@msg' => $e->getMessage()]);
      return NULL;
    }
  }

  public function getConfigurationForm() {
    $config = $this->getConfig();

    return [
      'ollama' => [
        '#type' => 'fieldset',
        '#title' => $this->t('Ollama Settings'),
        '#tree' => FALSE,
        'ollama_endpoint' => [
          '#type' => 'url',
          '#title' => $this->t('Ollama Endpoint'),
          '#default_value' => $config->get('ollama_endpoint') ?:
            'http://localhost:11434',
          '#required' => TRUE,
        ],
        'ollama_model' => [
          '#type' => 'textfield',
          '#title' => $this->t('Model Name'),
          '#default_value' => $config->get('ollama_model') ?: 'llama2',
          '#required' => TRUE,
        ],
      ],
    ];
  }

  public function isConfigured() {
    $config = $this->getConfig();
    return !empty($config->get('ollama_endpoint')) &&
      !empty($config->get('ollama_model'));
  }
}
// phpcs:enable
```

## Troubleshooting

### Plugin Not Appearing

1. Clear cache: `drush cr`
2. Check annotation syntax
3. Verify class namespace matches file location
4. Check for PHP syntax errors

### Configuration Not Saving

1. Verify field names in `getConfigurationForm()` match schema
2. Check `submitForm()` includes your fields
3. Clear configuration cache: `drush cr`

### API Errors

1. Check logs: ` / admin / reports / dblog`
2. Verify API credentials are correct
3. Test API endpoint with curl/Postman
4. Check timeout settings

## Additional Resources

- [Drupal Plugin API Documentation](
  https://www.drupal.org/docs/drupal-apis/plugin-api)
- [Form API Reference](https://api.drupal.org/api/drupal/elements)
- [Configuration Schema](
  https://www.drupal.org/docs/drupal-apis/configuration-api/
  configuration-schemametadata)
- [Guzzle HTTP Client](http://docs.guzzlephp.org/)

## Support

For questions or issues:
- Create an issue in the module's issue queue
- Review the VertexAi plugin implementation as a reference
- Check the module's README.md for general documentation
