# Commerce AI Suite Product Recommender

## Introduction

**Submodule of Commerce AI**

Provides personalized product recommendations using AI providers like Google
Vertex AI, OpenAI ChatGPT, or other AI services through a flexible plugin
system.

> **Note:** This is a submodule of the Commerce AI parent module. It depends
> on shared services provided by the parent module for GCS integration and
> other AI features. The module supports multiple AI providers through a
> plugin architecture, with Google Vertex AI included by default.

## Features

- **Pluggable AI Provider System**: Flexible plugin architecture supporting
  multiple AI providers (Google Vertex AI included by default)
- **AI-Powered Recommendations**: Leverages AI services for intelligent
  product recommendations based on user purchase history
- **Calculated Pricing**: Returns accurate prices with all Commerce
  adjustments (promotions, discounts, taxes) applied
- **Multi-Currency Support**: Includes currency codes for proper price
  formatting on frontend
- **Complete Product Data**: Returns SKU, title, URL, image, price, and
  currency in a single API call
- **GCS Integration**: Exports product catalog to Google Cloud Storage in
  JSONL format for grounded AI reasoning (when using compatible providers)
- **Queue-Based Export**: Uses Drupal's Queue API for efficient, scalable
  product data synchronization
- **User-Specific Context**: Analyzes order history to provide personalized
  recommendations for authenticated users
- **Fallback Recommendations**: Shows top sellers or featured products when
  AI provider is not configured or for users without purchase history
- **Block Integration**: Easy-to-place block for displaying recommendations
  anywhere on your site
- **REST API**: Full REST API endpoint for decoupled/headless implementations
- **Cron Automation**: Optional automated product sync via Drupal cron
- **Extensible**: Developers can add custom AI providers (see
  [DEVELOPER_GUIDE.md](DEVELOPER_GUIDE.md))

## Requirements

- **Commerce AI** (parent module) - provides shared GCS client service
- Drupal 10 or 11
- Commerce 2.x
- PHP 8.1 or higher

**Optional (required for specific AI providers):**
- For Google Vertex AI provider:
  - Google Cloud Platform account with:
    - Cloud Storage bucket
    - Vertex AI / Agentspace app configured
    - Service account with appropriate permissions
- For other AI providers: See provider-specific requirements in
  [DEVELOPER_GUIDE.md](DEVELOPER_GUIDE.md)

## Installation

1. **Install via Composer** (recommended):
   ```bash
   cd /path/to/drupal
   composer require google/cloud-storage:^1.30
   ```

2. **Enable the parent module first**:
   ```bash
   drush en commerce_ai_suite -y
   ```

3. **Enable this submodule**:
   ```bash
   drush en commerce_ai_suite_product_recommender -y
   drush cr
   ```

## Configuration

Navigate to: **Administration** → **Commerce** → **Configuration** →
**AI Product Recommender Settings**
(`/admin/commerce/config/ai-product-recommender`)

### 1. Select AI Provider

The module uses a plugin architecture to support multiple AI providers.

#### AI Provider Settings
- **Select AI Provider**: Choose your AI service from the dropdown:
  - **None**: No AI recommendations (uses fallback recommendations only)
  - **Google Vertex AI**: Uses Google's Vertex AI Discovery Engine (default)
  - **Custom Providers**: Additional providers can be added by developers
    (see [DEVELOPER_GUIDE.md](DEVELOPER_GUIDE.md))

When you select a provider, provider-specific configuration fields will
appear dynamically via AJAX.

#### Google Vertex AI Configuration

If you select "Google Vertex AI", the following fields will appear:

**Before configuring, set up your Google Cloud Platform resources:**

1. **Create a GCS Bucket**:
   ```bash
   gsutil mb -p YOUR_PROJECT_ID -l us-east4 gs://your-bucket-name
   ```

2. **Create a Service Account**:
   - Go to GCP Console → IAM & Admin → Service Accounts
   - Create a new service account
   - Grant the following roles:
     - Storage Object Admin (for GCS bucket)
     - Vertex AI User (for Vertex AI API)
   - Create and download a JSON key

3. **Configure Vertex AI Agentspace**:
   - Navigate to Vertex AI → Agent Builder
   - Create a new Agentspace app
   - Add a data source pointing to your GCS bucket
   - Configure the agent prompt (see below for recommended prompt)
   - Note your App ID

**Module Configuration Fields:**
- **GCP Project ID**: Your Google Cloud project ID
  (e.g., `acro-corp-ai-testing`)
- **Agentspace App ID**: Your Agentspace application ID
- **Agent Base Prompt**: The prompt template sent to the AI agent. Use
  `[HISTORY_JSON]` as a placeholder for user order history.

**Recommended Agent Prompt**:
```
You are an eCommerce expert. Using the product catalog, recommend 3-5
items based on user order history: [HISTORY_JSON]. Prioritize
compatibility and past purchase patterns. Output JSON:
{"recommendations": [{"id": "string", "title": "string",
"price": "number", "reason": "string"}], "confidence": 0-1}
```

### 2. Product Sync Settings (Optional)

Product sync is optional and only needed if your AI provider requires
product catalog data to be exported (e.g., Google Vertex AI).

#### Enable Automatic Sync
- **Enable automatic product sync via cron**: Check this box to enable
  automated product synchronization

When enabled, the following settings will appear dynamically via AJAX:

#### Google Cloud Storage Settings
- **GCP Project ID**: Your Google Cloud project ID
- **GCS Bucket Name**: The bucket where product data will be exported
- **Service Account Key**: Select the key from the Key module containing
  your service account JSON

#### Export Settings
- **Export Chunk Size**: Number of products per JSONL file (default: 50)

#### Cron Settings
- **Cron Interval (seconds)**: How often to sync (default: 900 = 15 minutes)

## Usage

### Initial Product Sync (Optional)

Product sync is only required if you're using an AI provider that needs access
to your product catalog (e.g., Google Vertex AI).

If you enabled automatic product sync in the configuration, perform an initial
sync:

```bash
drush commerce-ai-suite-product-recommender:sync-products
# or shorthand:
drush cairec-sync
```

This will queue all active products for export. Then run cron to process
the queue:

```bash
drush cron
```

If you selected "None" as your AI provider, product sync is not needed, and
the module will use fallback recommendations.

### Place the Recommendation Block

1. Navigate to: **Structure** → **Block layout**
2. Click "Place block" in your desired region
3. Find "AI Product Recommendations" block
4. Configure visibility settings (recommended: show only for authenticated
   users)
5. Save the block

The block will automatically:
- Show for authenticated users only
- Display personalized recommendations based on their order history
- Fall back to top sellers for users without purchase history
- Cache recommendations per user for 1 hour

### REST API Access

The module provides a REST API endpoint for external integrations:

**Endpoint:** `GET /api/recommendations`

**Authentication:** Requires authenticated user (cookie or basic auth)

**Response format:**
```json
{
  "recommendations": [
    {
      "id": "123",
      "sku": "PROD-123-XL",
      "title": "Product Name",
      "url": "/product/product-name",
      "image_url": "https://example.com/files/product-image.jpg",
      "price": "99.99",
      "currency_code": "USD",
      "reason": "Based on your previous purchases..."
    }
  ],
  "count": 3,
  "message": "No recommendations available at this time."
}
```

**Response fields:**
- `id`: Product entity ID
- `sku`: Product variation SKU (default variation)
- `title`: Product title
- `url`: Absolute URL to product page
- `image_url`: URL to product image (if available)
- `price`: Calculated price with adjustments (formatted to 2 decimals)
- `currency_code`: ISO currency code (e.g., "USD", "EUR", "GBP")
- `reason`: AI-generated explanation for the recommendation
- `count`: Total number of recommendations returned
- `message`: Status message (shown when no recommendations available)

For detailed API documentation, see [REST_API.md](REST_API.md)

**Caching:**
- Recommendations are cached per-user for 1 hour
- Cache is invalidated when products change or user places new orders
- Both empty and successful responses are cached

**Pricing Logic:**
The module returns calculated prices that include all Commerce adjustments:
1. First attempts to use the Commerce Price Calculator for adjusted prices
   (includes promotions, discounts, taxes)
2. Falls back to base product price if calculation fails
3. All prices are formatted to 2 decimal places
4. Currency code is determined from the current store context

### Automated Sync

If cron sync is enabled, products will automatically be re-exported based on
the configured interval. The module tracks the last sync time to avoid
redundant exports.

### Monitoring

Check Drupal logs for export status:

```bash
drush watchdog:show --type=commerce_ai_suite_product_recommender
```

## Product Data Structure

Products are exported to GCS in JSONL format with the following structure:

```json
{
  "id": "123",
  "content": {
    "title": "Product Name",
    "description": "Product description...",
    "sku": "PROD-SKU-001",
    "price": 99.99,
    "categories": ["Category 1", "Category 2"],
    "compatibility_tags": ["Compatible with X", "Works with Y"]
  },
  "metadata": {
    "source": "commerce_product",
    "product_id": "123",
    "product_type": "default",
    "created": "2024-01-15T10:30:00+00:00",
    "changed": "2024-01-20T14:45:00+00:00"
  }
}
```

## Customization

### Creating Custom AI Provider Plugins

The module's plugin architecture allows you to add support for additional AI
services (e.g., OpenAI ChatGPT, Anthropic Claude, Ollama, etc.) without
modifying the core module code.

For a complete guide on creating custom AI provider plugins, see
[DEVELOPER_GUIDE.md](DEVELOPER_GUIDE.md).

**Quick Overview:**

1. Create a plugin class in `src/Plugin/AiProvider/YourProvider.php`
2. Implement the `AiProviderInterface` (or extend `AiProviderBase`)
3. Add plugin annotation with metadata (id, label, description)
4. Implement `getRecommendations()` to communicate with your AI service
5. Implement `getConfigurationForm()` to provide settings fields
6. Clear cache: `drush cr`

Your custom provider will automatically appear in the "Select AI Provider"
dropdown.

### Custom Product Fields

To include additional product fields in the export, extend
`ProductExporterService::transformProductToDocument()`:

```php
// In a custom module's service decorator or event subscriber:
protected function transformProductToDocument($product) {
  $document = parent::transformProductToDocument($product);

  // Add custom field
  if ($product->hasField('field_custom_attribute')) {
    $document['content']['custom_attribute'] =
      $product->get('field_custom_attribute')->value;
  }

  return $document;
}
```

### Custom Recommendation Logic

To customize the recommendation service behavior, create a service decorator:

```yaml
# mymodule.services.yml
services:
  mymodule.recommendation_service_decorator:
    class: Drupal\mymodule\Service\CustomRecommendationService
    decorates: commerce_ai_suite_product_recommender.recommendation_service
    arguments: ['@mymodule.recommendation_service_decorator.inner']
```

### Theming

Override the block template by copying
`commerce-ai-suite-product-recommendation-block.html.twig` to your theme:

```
themes/custom/YOUR_THEME/templates/
  commerce-ai-suite-product-recommendation-block.html.twig
```

Add custom CSS:

```css
.ai-product-recommendations {
  display: grid;
  grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
  gap: 1rem;
}

.ai-product-recommendations__reason {
  font-style: italic;
  color: #666;
  margin-top: 0.5rem;
}
```

## Troubleshooting

### Products not appearing in recommendations

1. **Check logs**:
   `drush watchdog:show --type=commerce_ai_suite_product_recommender`
2. **Verify GCS upload**: Use Cloud Console or
   `gsutil ls gs://your-bucket-name/products/`
3. **Test Agentspace**: Query the Agentspace API directly to verify it can
   access the data
4. **Clear cache**: `drush cr`

### "Service account key is invalid" error

Ensure the JSON key is:
- Complete (not truncated)
- Valid JSON format
- Of type "service_account"
- Has the correct permissions

### Slow recommendation loading

- Reduce the number of products per chunk
- Increase cache max-age in the block
- Optimize Vertex AI endpoint configuration
- Consider implementing a warmup cache strategy

## Security Considerations

- **Service Account Key**: Stored securely using the Key module
- **API Keys**: Never commit to version control; use environment variables
  or Key module
- **User Data**: Order history is anonymized when sent to Vertex AI (only
  product IDs and categories)
- **Access Control**: Block visibility restricted by default to
  authenticated users
- **Rate Limiting**: Consider implementing rate limiting for Vertex AI API
  calls

## Performance

- **Queue Processing**: Uses Drupal's Queue API for efficient bulk exports
- **Chunked Uploads**: Products exported in configurable chunks to avoid
  memory issues
- **Caching**: Recommendations cached per-user for 1 hour with cache tags for
  automatic invalidation
- **Calculated Pricing**: Price calculation happens only once per request and
  is cached with the recommendation
- **Single API Call**: Frontend gets all product data (SKU, title, URL,
  image, price, currency) in one request
- **Retry Logic**: GCS uploads include automatic retry with exponential
  backoff

## Extending the Module

### Adding New AI Providers

The module is designed to be extended with new AI provider plugins. Developers
can add support for:

- **OpenAI ChatGPT**: See example implementation in
  [DEVELOPER_GUIDE.md](DEVELOPER_GUIDE.md)
- **Anthropic Claude**: Similar pattern to ChatGPT example
- **Ollama**: For local AI inference (example included in developer guide)
- **Custom/Proprietary AI services**: Any service with an HTTP API

### Other Expansion Ideas

- Add support for Elasticsearch integration for faster local queries
- Implement A/B testing for recommendation strategies
- Add analytics tracking for recommendation click-through rates
- Support for cross-selling and up-selling specific use cases
- Real-time product updates via webhooks instead of batch sync
- Integration with recommendation analytics services

## Contributing

This module was developed for Concept2 by Acro Commerce. For issues,
feature requests, or contributions, please contact the development team.

## Credits

- **Developed by**: Acro Commerce
- **Client**: Concept2
- **Technology**: Google Vertex AI, Gemini Enterprise, Drupal Commerce

## License

GPL-2.0-or-later
