# Product Recommendation REST API

This document describes the REST API endpoint for accessing AI product
recommendations from external applications (e.g., React frontend).

## Endpoint

```
GET /api/recommendations
```

## Authentication

The API requires authentication. Use one of the following methods:

### Cookie Authentication (Recommended for same-domain)
If your frontend is on the same domain, you can use cookie-based
authentication after the user logs in.

### Basic Authentication
For cross-domain requests, use HTTP Basic Auth:

```bash
curl -X GET https://example.com/api/recommendations \
  -H "Authorization: Basic BASE64_ENCODED_CREDENTIALS" \
  -H "Accept: application/json"
```

## Request Example

### Using fetch (JavaScript)

```javascript
fetch('https://your-drupal-site.com/api/recommendations', {
  method: 'GET',
  headers: {
    'Accept': 'application/json',
    'Content-Type': 'application/json',
  },
  credentials: 'include', // Include cookies
})
  .then(response => response.json())
  .then(data => {
    console.log('Recommendations:', data.recommendations);
  })
  .catch(error => {
    console.error('Error fetching recommendations:', error);
  });
```

### Using axios (JavaScript)

```javascript
import axios from 'axios';

axios.get('https://your-drupal-site.com/api/recommendations', {
  withCredentials: true,
  headers: {
    'Accept': 'application/json',
  },
})
  .then(response => {
    console.log('Recommendations:', response.data.recommendations);
  })
  .catch(error => {
    console.error('Error:', error);
  });
```

## Response Format

### Success Response (200 OK)

```json
{
  "recommendations": [
    {
      "id": "123",
      "title": "Product Name",
      "price": "99.99",
      "reason": "Based on your recent purchase of..."
    },
    {
      "id": "456",
      "title": "Another Product",
      "price": "149.99",
      "reason": "Complements your training equipment"
    }
  ],
  "count": 2
}
```

### Empty Recommendations (200 OK)

```json
{
  "recommendations": [],
  "message": "No recommendations available at this time."
}
```

### Error Responses

#### 403 Forbidden
User doesn't have permission to access the endpoint.

```json
{
  "message": "Access denied."
}
```

## CORS Configuration

If your frontend is hosted on a different domain, you'll need to
configure CORS in Drupal.

### Option 1: services.yml (Recommended)

Add to `sites/default/services.yml`:

```yaml
cors.config:
  enabled: true
  allowedOrigins:
    - 'https://your-frontend-domain.com'
  allowedMethods:
    - GET
    - POST
    - OPTIONS
  allowedHeaders:
    - 'Content-Type'
    - 'Authorization'
    - 'X-Requested-With'
  exposedHeaders: false
  maxAge: 1000
  supportsCredentials: true
```

### Option 2: .htaccess (Apache)

Add to `.htaccess`:

```apache
<IfModule mod_headers.c>
  Header set Access-Control-Allow-Origin "https://your-frontend.com"
  Header set Access-Control-Allow-Credentials "true"
  Header set Access-Control-Allow-Methods "GET, OPTIONS"
  Header set Access-Control-Allow-Headers "Content-Type, Authorization"
</IfModule>
```

## Integration Example (React)

```jsx
import React, { useEffect, useState } from 'react';

function ProductRecommendations() {
  const [recommendations, setRecommendations] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);

  useEffect(() => {
    fetch('https://your-drupal.com/api/recommendations', {
      credentials: 'include',
      headers: {
        'Accept': 'application/json',
      },
    })
      .then(response => {
        if (!response.ok) {
          throw new Error('Failed to fetch recommendations');
        }
        return response.json();
      })
      .then(data => {
        setRecommendations(data.recommendations);
        setLoading(false);
      })
      .catch(err => {
        setError(err.message);
        setLoading(false);
      });
  }, []);

  if (loading) return <div>Loading recommendations...</div>;
  if (error) return <div>Error: {error}</div>;

  return (
    <div className="recommendations">
      <h2>Recommended for You</h2>
      {recommendations.length === 0 ? (
        <p>No recommendations available.</p>
      ) : (
        <div className="product-grid">
          {recommendations.map(product => (
            <div key={product.id} className="product-card">
              <h3>{product.title}</h3>
              <p className="price">${product.price}</p>
              <p className="reason">{product.reason}</p>
            </div>
          ))}
        </div>
      )}
    </div>
  );
}

export default ProductRecommendations;
```

## Security Considerations

1. **HTTPS Only**: Always use HTTPS in production
2. **Authentication**: Ensure users are properly authenticated
3. **Rate Limiting**: Consider implementing rate limiting to prevent abuse
4. **CORS**: Restrict allowed origins to your specific frontend domain(s)
5. **Credentials**: Never expose API keys or credentials to the frontend

## Testing

Test the endpoint using curl:

```bash
# Test with authentication
curl -X GET https://your-site.com/api/recommendations \
  -H "Accept: application/json" \
  -u "username:password"
```

## Troubleshooting

### Issue: CORS errors in browser console

**Solution**: Configure CORS as described above

### Issue: 401 Unauthorized

**Solution**: Ensure the user is logged in and credentials are being sent

### Issue: Empty recommendations

**Possible causes**:
- User has no order history
- AI service is not configured
- Products haven't been exported to GCS yet

**Solution**: Check Drupal logs and ensure the AI service is configured at
`/admin/commerce/config/ai-product-recommender`
