<?php

/**
 * @file
 * Hooks provided by the Commerce AI Suite Product Recommender module.
 */

/**
 * @addtogroup hooks
 * @{
 */

/**
 * Alter the product document before exporting to GCS.
 *
 * This hook allows modules to modify the product document structure
 * before it is exported to Google Cloud Storage for AI processing.
 *
 * @param array &$document
 *   The product document array to be altered. Contains:
 *   - id: The product ID.
 *   - content: Array of product data including:
 *     - title: Product title.
 *     - description: Product description.
 *     - sku: Default SKU.
 *     - price: Default price data.
 *     - categories: Array of category data.
 *     - variations: Array of variation data.
 *     - variation_count: Number of variations.
 *   - metadata: Array of metadata including:
 *     - source: Always 'commerce_product'.
 *     - product_id: The product ID.
 *     - product_type: Product bundle.
 *     - created: ISO 8601 creation date.
 *     - changed: ISO 8601 last modified date.
 *     - status: 'published' or 'unpublished'.
 * @param \Drupal\commerce_product\Entity\ProductInterface $product
 *   The product entity being transformed.
 *
 * @see \Drupal\commerce_ai_suite_product_recommender\Service\ProductExporterService::transformProductToDocument()
 */
function hook_commerce_ai_suite_product_recommender_document_alter(
  array &$document,
  $product,
) {
  // Example: Add custom field data to the document.
  if ($product->hasField('field_custom_data')) {
    $document['content']['custom_data'] = $product->get(
      'field_custom_data'
    )->value;
  }

  // Example: Add custom metadata.
  $document['metadata']['custom_flag'] = TRUE;

  // Example: Modify existing content.
  if (isset($document['content']['description'])) {
    // Append additional information to description.
    $document['content']['description'] .= ' [Custom Info]';
  }

  // Example: Filter variations based on custom logic.
  if (isset($document['content']['variations'])) {
    $document['content']['variations'] = array_filter(
      $document['content']['variations'],
      function ($variation) {
        // Only include variations with price > 10.
        return isset($variation['price']['number']) &&
          $variation['price']['number'] > 10;
      }
    );
    // Update variation count.
    $document['content']['variation_count'] = count(
      $document['content']['variations']
    );
  }
}

/**
 * Alter product recommendation data before rendering.
 *
 * This hook allows modules to modify the recommendation data array
 * before it is passed to the template for rendering. This is useful
 * for customizing image fields, URLs, or adding additional data.
 *
 * @param array &$product_data
 *   An array of product recommendation data. Each item contains:
 *   - id: The product ID.
 *   - title: Product title.
 *   - url: Product URL.
 *   - image_url: Product image URL (can be modified).
 *   - price: Product price.
 *   - reason: AI recommendation reason.
 *   - product: The full product entity for additional data access.
 *
 * @see \Drupal\commerce_ai_suite_product_recommender\Plugin\Block\ProductRecommendationBlock::build()
 */
function hook_commerce_ai_suite_product_recommender_recommendation_data_alter(
  array &$product_data,
) {
  // Example: Use a different image field.
  foreach ($product_data as &$item) {
    $product = $item['product'];
    if ($product->hasField('field_representative_image')) {
      $image_field = $product->get('field_representative_image')->first();
      if ($image_field && $image_field->entity) {
        $item['image_url'] = $image_field->entity->createFileUrl();
      }
    }
  }

  // Example: Add a badge for featured products.
  foreach ($product_data as &$item) {
    $product = $item['product'];
    if ($product->hasField('field_featured') && $product->get('field_featured')->value) {
      $item['badge'] = 'Featured';
    }
  }
}

/**
 * @} End of "addtogroup hooks".
 */
