<?php

namespace Drupal\commerce_ai_suite_product_recommender\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderManager;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Commerce AI Suite Product Recommender settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * The AI provider manager.
   *
   * @var \Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderManager
   */
  protected $aiProviderManager;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository.
   * @param \Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderManager $ai_provider_manager
   *   The AI provider manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    KeyRepositoryInterface $key_repository,
    AiProviderManager $ai_provider_manager,
  ) {
    parent::__construct($config_factory);
    $this->keyRepository = $key_repository;
    $this->aiProviderManager = $ai_provider_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('commerce_ai_suite_product_recommender.ai_provider_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['commerce_ai_suite_product_recommender.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'commerce_ai_suite_product_recommender_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('commerce_ai_suite_product_recommender.settings');

    // AI Provider section.
    $form['ai_provider_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('AI Provider'),
      '#open' => TRUE,
      '#description' => $this->t('Configure the AI service used for generating product recommendations.'),
    ];

    // Add "None" option to provider options.
    $provider_options = ['' => $this->t('- None -')] + $this->aiProviderManager->getProviderOptions();

    $form['ai_provider_settings']['ai_provider'] = [
      '#type' => 'select',
      '#title' => $this->t('Select AI Provider'),
      '#options' => $provider_options,
      '#default_value' => $config->get('ai_provider') ?: '',
      '#description' => $this->t('Choose the AI service to use for product recommendations. Select a provider to see configuration options.'),
      '#ajax' => [
        'callback' => '::updateProviderFields',
        'wrapper' => 'provider-specific-fields',
      ],
    ];

    // Container for provider-specific fields.
    $form['ai_provider_settings']['provider_specific_fields'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'provider-specific-fields'],
    ];

    // Get the selected provider (from AJAX or default).
    $selected_provider = $form_state->getValue('ai_provider') ?? $config->get('ai_provider');

    // Only show provider fields if a provider is selected.
    if (!empty($selected_provider)) {
      try {
        $provider = $this->aiProviderManager->createInstance($selected_provider);
        $provider_form = $provider->getConfigurationForm();
        $form['ai_provider_settings']['provider_specific_fields'] = array_merge(
          $form['ai_provider_settings']['provider_specific_fields'],
          $provider_form
        );
      }
      catch (\Exception $e) {
        $this->messenger()->addError($this->t('Error loading provider configuration: @message', ['@message' => $e->getMessage()]));
      }
    }

    // Product Sync section.
    $form['product_sync'] = [
      '#type' => 'details',
      '#title' => $this->t('Product Sync'),
      '#open' => TRUE,
      '#description' => $this->t('Configure how product data is synced to your AI provider.'),
    ];

    $form['product_sync']['cron_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic product sync via cron'),
      '#default_value' => $config->get('cron_enabled'),
      '#description' => $this->t('When enabled, products will be automatically exported to Google Cloud Storage on a regular schedule.'),
      '#ajax' => [
        'callback' => '::updateSyncFields',
        'wrapper' => 'sync-settings-wrapper',
      ],
    ];

    // Container for sync settings (only shown when cron is enabled).
    $form['product_sync']['sync_settings'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'sync-settings-wrapper'],
    ];

    // Get the cron_enabled value (from AJAX or default).
    $cron_enabled = $form_state->getValue('cron_enabled') ?? $config->get('cron_enabled');

    // Only show sync settings if cron is enabled.
    if ($cron_enabled) {
      $form['product_sync']['sync_settings']['gcs'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Google Cloud Storage Settings'),
        '#description' => $this->t('Configure Google Cloud Storage for product data export. <a href="@gcs_url" target="_blank">Learn more about GCS</a>', ['@gcs_url' => 'https://cloud.google.com/storage/docs']),
      ];

      $form['product_sync']['sync_settings']['gcs']['gcp_project_id'] = [
        '#type' => 'textfield',
        '#title' => $this->t('GCP Project ID'),
        '#default_value' => $config->get('gcp_project_id'),
        '#required' => TRUE,
        '#description' => $this->t('Your Google Cloud Platform project ID. Find this in the <a href="@console" target="_blank">GCP Console</a> dashboard.', ['@console' => 'https://console.cloud.google.com']),
      ];

      $form['product_sync']['sync_settings']['gcs']['gcs_bucket_name'] = [
        '#type' => 'textfield',
        '#title' => $this->t('GCS Bucket Name'),
        '#default_value' => $config->get('gcs_bucket_name'),
        '#required' => TRUE,
        '#description' => $this->t('The GCS bucket where product data will be exported. <a href="@create" target="_blank">Create a bucket</a>', ['@create' => 'https://console.cloud.google.com/storage/browser']),
      ];

      $form['product_sync']['sync_settings']['gcs']['gcs_service_account_key'] = [
        '#type' => 'key_select',
        '#title' => $this->t('Service Account Key'),
        '#default_value' => $config->get('gcs_service_account_key'),
        '#required' => TRUE,
        '#description' => $this->t('Select the key containing the GCS service account JSON. Manage keys at <a href="@keys">Keys</a>. <a href="@sa" target="_blank">Create service account</a>', [
          '@keys' => '/admin/config/system/keys',
          '@sa' => 'https://console.cloud.google.com/iam-admin/serviceaccounts',
        ]),
      ];

      $form['product_sync']['sync_settings']['export'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Export Settings'),
      ];

      $form['product_sync']['sync_settings']['export']['export_chunk_size'] = [
        '#type' => 'number',
        '#title' => $this->t('Export Chunk Size'),
        '#default_value' => $config->get('export_chunk_size'),
        '#required' => TRUE,
        '#min' => 1,
        '#max' => 500,
        '#description' => $this->t(
          'Number of products per JSONL file chunk.'
        ),
      ];

      $form['product_sync']['sync_settings']['cron'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Cron Schedule'),
      ];

      $form['product_sync']['sync_settings']['cron']['cron_interval'] = [
        '#type' => 'number',
        '#title' => $this->t('Cron Interval (seconds)'),
        '#default_value' => $config->get('cron_interval'),
        '#required' => TRUE,
        '#min' => 60,
        '#description' => $this->t('How often to sync products (minimum 60 seconds). Default: @default (@minutes minutes).', [
          '@default' => '900',
          '@minutes' => '15',
        ]),
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback to update provider-specific fields.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The provider-specific fields element.
   */
  public function updateProviderFields(array &$form, FormStateInterface $form_state) {
    return $form['ai_provider_settings']['provider_specific_fields'];
  }

  /**
   * AJAX callback to update sync settings fields.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The sync settings element.
   */
  public function updateSyncFields(array &$form, FormStateInterface $form_state) {
    return $form['product_sync']['sync_settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(
    array &$form,
    FormStateInterface $form_state,
  ) {
    parent::validateForm($form, $form_state);

    // Validate that the selected key contains valid JSON.
    $key_id = $form_state->getValue('gcs_service_account_key');
    if (!empty($key_id)) {
      $key = $this->keyRepository->getKey($key_id);

      if (!$key) {
        $form_state->setErrorByName(
          'gcs_service_account_key',
          $this->t('The selected key could not be found.')
        );
        return;
      }

      $key_value = $key->getKeyValue();
      if (empty($key_value)) {
        $form_state->setErrorByName(
          'gcs_service_account_key',
          $this->t('The selected key is empty.')
        );
        return;
      }

      $decoded = json_decode($key_value, TRUE);
      if (json_last_error() !== JSON_ERROR_NONE) {
        $form_state->setErrorByName(
          'gcs_service_account_key',
          $this->t('The selected key does not contain valid JSON.')
        );
        return;
      }

      if (empty($decoded['type']) ||
        $decoded['type'] !== 'service_account') {
        $form_state->setErrorByName(
          'gcs_service_account_key',
          $this->t('The key must contain a service account JSON with type "service_account".')
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('commerce_ai_suite_product_recommender.settings');
    $config
      ->set('ai_provider', $form_state->getValue('ai_provider'))
      ->set('gcp_project_id', $form_state->getValue('gcp_project_id'))
      ->set('gcs_bucket_name', $form_state->getValue('gcs_bucket_name'))
      ->set(
        'gcs_service_account_key',
        $form_state->getValue('gcs_service_account_key')
      )
      ->set(
        'vertex_ai_endpoint',
        $form_state->getValue('vertex_ai_endpoint')
      )
      ->set(
        'vertex_ai_location',
        $form_state->getValue('vertex_ai_location')
      )
      ->set(
        'agentspace_app_id',
        $form_state->getValue('agentspace_app_id')
      )
      ->set(
        'agentspace_config_id',
        $form_state->getValue('agentspace_config_id')
      )
      ->set(
        'agentspace_api_key',
        $form_state->getValue('agentspace_api_key')
      )
      ->set('agent_prompt', $form_state->getValue('agent_prompt'))
      ->set(
        'export_chunk_size',
        $form_state->getValue('export_chunk_size')
      )
      ->set('cron_enabled', $form_state->getValue('cron_enabled'))
      ->set('cron_interval', $form_state->getValue('cron_interval'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
