<?php

namespace Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Manages discovery and instantiation of AI provider plugins.
 */
class AiProviderManager extends DefaultPluginManager {

  /**
   * Constructs a new AiProviderManager.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   The cache backend.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct(
      'Plugin/AiProvider',
      $namespaces,
      $module_handler,
      'Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderInterface',
      'Drupal\commerce_ai_suite_product_recommender\Annotation\AiProvider'
    );

    $this->alterInfo('commerce_ai_suite_provider_info');
    $this->setCacheBackend($cache_backend, 'commerce_ai_suite_provider_plugins');
  }

  /**
   * Gets available AI providers sorted by weight.
   *
   * @return array
   *   Array of plugin definitions keyed by plugin ID.
   */
  public function getAvailableProviders() {
    $definitions = $this->getDefinitions();

    // Sort by weight.
    uasort($definitions, function ($a, $b) {
      $a_weight = $a['weight'] ?? 0;
      $b_weight = $b['weight'] ?? 0;
      return $a_weight <=> $b_weight;
    });

    return $definitions;
  }

  /**
   * Gets provider options for form select.
   *
   * @return array
   *   Array of provider labels keyed by plugin ID.
   */
  public function getProviderOptions() {
    $options = [];
    foreach ($this->getAvailableProviders() as $plugin_id => $definition) {
      $options[$plugin_id] = $definition['label'];
    }
    return $options;
  }

}
