<?php

namespace Drupal\commerce_ai_suite_product_recommender\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\commerce_ai_suite_product_recommender\Service\RecommendationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an 'AI Product Recommendation' block.
 *
 * @Block(
 *   id = "commerce_ai_suite_product_recommender_recommendation_block",
 *   admin_label = @Translation("AI Product Recommendations"),
 *   category = @Translation("Commerce")
 * )
 */
class ProductRecommendationBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The recommendation service.
   *
   * @var \Drupal\commerce_ai_suite_product_recommender\Service\RecommendationService
   */
  protected $recommendationService;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a ProductRecommendationBlock object.
   *
   * @param array $configuration
   *   A configuration array with info about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\commerce_ai_suite_product_recommender\Service\RecommendationService $recommendation_service
   *   The recommendation service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    RecommendationService $recommendation_service,
    AccountProxyInterface $current_user,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->recommendationService = $recommendation_service;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get(
        'commerce_ai_suite_product_recommender.recommendation_service'
      ),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    // Only show for authenticated users.
    if (!$this->currentUser->isAuthenticated()) {
      return [];
    }

    // Get AI recommendations.
    $recommendations = $this->recommendationService
      ->getRecommendationsForCurrentUser();

    if (empty($recommendations)) {
      return [
        '#markup' => $this->t(
          'No recommendations available at this time.'
        ),
      ];
    }

    // Enrich with full product data (URLs, images, etc.).
    $product_data = $this->recommendationService
      ->enrichRecommendationsWithProductData($recommendations);

    if (empty($product_data)) {
      return [
        '#markup' => $this->t(
          'No recommendations available at this time.'
        ),
      ];
    }

    return [
      '#theme' => 'commerce_ai_suite_product_recommender_recommendation_block',
      '#recommendations' => $product_data,
      '#attached' => [
        'library' => [
          'commerce_ai_suite_product_recommender/recommendation_block',
        ],
      ],
      '#cache' => [
        'contexts' => ['user'],
        'max-age' => 3600,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return Cache::mergeContexts(parent::getCacheContexts(), ['user']);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $tags = parent::getCacheTags();
    // Invalidate when any product changes.
    $tags = Cache::mergeTags($tags, ['commerce_product_list']);
    return $tags;
  }

}
