<?php

namespace Drupal\commerce_ai_suite_product_recommender\Plugin\rest\resource;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\commerce_ai_suite_product_recommender\Service\RecommendationService;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

/**
 * Provides a Product Recommendation Resource.
 *
 * @RestResource(
 *   id = "commerce_ai_suite_product_recommender_recommendation",
 *   label = @Translation("Product Recommendation"),
 *   uri_paths = {
 *     "canonical" = "/api/product-recommendations"
 *   },
 *   authentication_types = {
 *     "cookie",
 *     "basic_auth",
 *   },
 *   authentication_optional = FALSE
 * )
 */
class ProductRecommendationResource extends ResourceBase {

  /**
   * The recommendation service.
   *
   * @var \Drupal\commerce_ai_suite_product_recommender\Service\RecommendationService
   */
  protected $recommendationService;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a ProductRecommendationResource object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param array $serializer_formats
   *   The available serialization formats.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   * @param \Drupal\commerce_ai_suite_product_recommender\Service\RecommendationService $recommendation_service
   *   The recommendation service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    RecommendationService $recommendation_service,
    AccountProxyInterface $current_user,
  ) {
    parent::__construct(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $serializer_formats,
      $logger
    );
    $this->recommendationService = $recommendation_service;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition,
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')
        ->get('commerce_ai_suite_product_recommender'),
      $container->get(
        'commerce_ai_suite_product_recommender.recommendation_service'
      ),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function access($operation = 'view', $optional = NULL) {
    if ($operation === 'get') {
      return AccessResult::allowedIf($this->currentUser->isAuthenticated())
        ->addCacheContexts(['user']);
    }
    return parent::access($operation, $optional);
  }

  /**
   * Responds to GET requests.
   *
   * Returns AI product recommendations for the current user with full
   * product details (title, price, URL, image).
   *
   * @return \Drupal\rest\ResourceResponse
   *   The response containing enriched recommendations.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException
   *   Throws exception if user is not authenticated.
   */
  public function get() {
    // Explicit auth check (redundant with access() but defensive).
    if (!$this->currentUser->isAuthenticated()) {
      throw new AccessDeniedHttpException(
        'Authentication required for recommendations.'
      );
    }

    // Get AI recommendations.
    $recommendations = $this->recommendationService
      ->getRecommendationsForCurrentUser();

    $product_data = [];
    $message = 'No recommendations available at this time.';
    $count = 0;

    if (!empty($recommendations)) {
      // Enrich with full product data (URLs, images, etc.).
      $product_data = $this->recommendationService
        ->enrichRecommendationsWithProductData($recommendations);
      $count = count($product_data);
      if (empty($product_data)) {
        $message = 'Recommendations generated, but product data unavailable.';
      }
    }

    $response_data = [
      'recommendations' => $product_data,
      'count' => $count,
      'message' => $message,
    ];

    $response = new ResourceResponse($response_data);

    // Unified caching: 1 hour TTL, per-user variation, invalidate on
    // user/orders or product lists. Use CacheableMetadata for proper bubbling
    // and granularity.
    $cache_meta = CacheableMetadata::createFromRenderArray([
      '#cache' => [
      // Varies by authenticated user.
        'contexts' => ['user'],
      // 1 hour TTL for all responses (unified from 300s).
        'max-age' => 3600,
        'tags' => [
          // Broad product list for catalog changes.
          'commerce_product_list',
          // User-specific for order history updates (e.g., new purchases
          // invalidate).
          'user:' . $this->currentUser->id(),
        ],
      ],
    ]);
    $response->addCacheableDependency($cache_meta);

    // Optional: Add HTTP headers for proxy/CDN caching.
    $response->headers->set('Cache-Control', 'max-age=3600, private');
    $response->headers->set('Vary', 'Cookie, Authorization');

    return $response;
  }

}
