<?php

namespace Drupal\commerce_ai_suite_product_recommender\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\commerce\Context;
use Drupal\commerce_ai_suite\Service\GcsClientService;
use Drupal\commerce_order\PriceCalculatorInterface;
use Drupal\commerce_price\Resolver\ChainPriceResolverInterface;
use Drupal\commerce_store\CurrentStoreInterface;

/**
 * Service for exporting product data to GCS.
 */
class ProductExporterService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The GCS client service.
   *
   * @var \Drupal\commerce_ai_suite\Service\GcsClientService
   */
  protected $gcsClient;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The current store service.
   *
   * @var \Drupal\commerce_store\CurrentStoreInterface
   */
  protected $currentStore;

  /**
   * The chain price resolver.
   *
   * @var \Drupal\commerce_price\Resolver\ChainPriceResolverInterface
   */
  protected $chainPriceResolver;

  /**
   * The price calculator.
   *
   * @var \Drupal\commerce_order\PriceCalculatorInterface
   */
  protected $priceCalculator;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a ProductExporterService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\commerce_ai_suite\Service\GcsClientService $gcs_client
   *   The GCS client service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Drupal\commerce_store\CurrentStoreInterface $current_store
   *   The current store service.
   * @param \Drupal\commerce_price\Resolver\ChainPriceResolverInterface $chain_price_resolver
   *   The chain price resolver.
   * @param \Drupal\commerce_order\PriceCalculatorInterface $price_calculator
   *   The price calculator.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
    GcsClientService $gcs_client,
    LoggerChannelFactoryInterface $logger_factory,
    QueueFactory $queue_factory,
    CurrentStoreInterface $current_store,
    ChainPriceResolverInterface $chain_price_resolver,
    PriceCalculatorInterface $price_calculator,
    AccountProxyInterface $current_user,
    ModuleHandlerInterface $module_handler,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->gcsClient = $gcs_client;
    $channel = 'commerce_ai_suite_product_recommender';
    $this->logger = $logger_factory->get($channel);
    $this->queueFactory = $queue_factory;
    $this->currentStore = $current_store;
    $this->chainPriceResolver = $chain_price_resolver;
    $this->priceCalculator = $price_calculator;
    $this->currentUser = $current_user;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Queues all active products for export.
   *
   * @return int
   *   The number of products queued.
   */
  public function queueProductsForExport() {
    try {
      $storage = $this->entityTypeManager->getStorage('commerce_product');

      // Load all published products.
      $query = $storage->getQuery()
        ->condition('status', 1)
        ->accessCheck(FALSE);

      $product_ids = $query->execute();

      if (empty($product_ids)) {
        $this->logger->info('No products found to export.');
        return 0;
      }

      $config_name = 'commerce_ai_suite_product_recommender.settings';
      $config = $this->configFactory->get($config_name);
      $chunk_size = $config->get('export_chunk_size') ?: 50;

      // Split products into chunks.
      $chunks = array_chunk($product_ids, $chunk_size, TRUE);
      $queue_name = 'commerce_ai_suite_product_recommender_export';
      $queue = $this->queueFactory->get($queue_name);

      foreach ($chunks as $index => $chunk) {
        $queue->createItem([
          'chunk_index' => $index,
          'product_ids' => $chunk,
        ]);
      }

      $this->logger->info(
        'Queued @count products in @chunks chunks for export.',
        [
          '@count' => count($product_ids),
          '@chunks' => count($chunks),
        ]
      );

      return count($product_ids);
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to queue products for export: @message',
        ['@message' => $e->getMessage()]
      );
      return 0;
    }
  }

  /**
   * Processes a chunk of products and exports to GCS.
   *
   * @param array $product_ids
   *   Array of product IDs to export.
   * @param int $chunk_index
   *   The chunk index for file naming.
   *
   * @return bool
   *   TRUE if export succeeded, FALSE otherwise.
   */
  public function exportProductChunk(array $product_ids, $chunk_index) {
    try {
      $storage = $this->entityTypeManager->getStorage('commerce_product');
      $products = $storage->loadMultiple($product_ids);

      if (empty($products)) {
        $this->logger->warning(
          'No products found for chunk @index',
          ['@index' => $chunk_index]
        );
        return FALSE;
      }

      $documents = [];

      foreach ($products as $product) {
        // Double-check product is published.
        if ($product->get('status')->value != 1) {
          continue;
        }

        $document = $this->transformProductToDocument($product);
        if ($document) {
          $documents[] = json_encode($document);
        }
      }

      if (empty($documents)) {
        $this->logger->warning(
          'No valid documents generated for chunk @index',
          ['@index' => $chunk_index]
        );
        return FALSE;
      }

      // Create JSONL content.
      $jsonl_content = implode("\n", $documents);

      // Upload to GCS.
      $config_name = 'commerce_ai_suite_product_recommender.settings';
      $config = $this->configFactory->get($config_name);
      $bucket_name = $config->get('gcs_bucket_name');

      // Files are placed in the root of the bucket and overwrite on each sync.
      $file_name = sprintf('chunk_%04d.jsonl', $chunk_index);

      $result = $this->gcsClient->uploadFileWithRetry(
        $config_name,
        $bucket_name,
        $jsonl_content,
        $file_name
      );

      if ($result) {
        $this->logger->info(
          'Successfully exported chunk @index with @count products.',
          [
            '@index' => $chunk_index,
            '@count' => count($documents),
          ]
        );
      }

      return $result;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to export product chunk @index: @message',
        [
          '@index' => $chunk_index,
          '@message' => $e->getMessage(),
        ]
      );
      return FALSE;
    }
  }

  /**
   * Transforms a product entity into a document for RAG.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   *
   * @return array|null
   *   The document array or NULL if transformation fails.
   */
  protected function transformProductToDocument($product) {
    try {
      $enabled_variations = $this->getEnabledVariations($product);
      $description = $this->extractProductDescription($product);
      $categories = $this->extractCategories($product);
      [$default_sku, $default_price] = $this->getDefaultVariationData(
        $product
      );

      $document = [
        'id' => $product->id(),
        'content' => [
          'title' => $product->label(),
          'description' => $description,
          'sku' => $default_sku,
          'price' => $default_price,
          'categories' => $categories,
          'variations' => $enabled_variations,
          'variation_count' => count($enabled_variations),
        ],
        'metadata' => $this->buildProductMetadata($product),
      ];

      // Allow other modules to alter the product document.
      $this->moduleHandler->alter(
        'commerce_ai_suite_product_recommender_document',
        $document,
        $product
      );

      return $document;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to transform product @id: @message',
        [
          '@id' => $product->id(),
          '@message' => $e->getMessage(),
        ]
      );
      return NULL;
    }
  }

  /**
   * Extracts enabled variations from a product.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   *
   * @return array
   *   Array of variation data.
   */
  protected function getEnabledVariations($product) {
    $variations = $product->getVariations();
    $enabled_variations = [];

    foreach ($variations as $variation) {
      // Check if variation is published.
      if ($variation->get('status')->value != 1) {
        continue;
      }

      $variation_data = [
        'variation_id' => $variation->id(),
        'sku' => $variation->getSku(),
        'title' => $variation->label(),
      ];

      // Add price data.
      $variation_data['price'] = $this->extractVariationPrice($variation);
      $variation_data['calculated_price'] =
        $this->extractCalculatedPrice($variation);
      $variation_data['resolved_price'] =
        $this->extractResolvedPrice($variation);

      $enabled_variations[] = $variation_data;
    }

    return $enabled_variations;
  }

  /**
   * Extracts base price from a variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   *
   * @return array|null
   *   Price data or NULL.
   */
  protected function extractVariationPrice($variation) {
    if (!$variation->hasField('price')) {
      return NULL;
    }

    $price_field = $variation->get('price')->first();
    if (!$price_field || !method_exists($price_field, 'toPrice')) {
      return NULL;
    }

    $price = $price_field->toPrice();
    return [
      'number' => $price->getNumber(),
      'currency_code' => $price->getCurrencyCode(),
      'formatted' => $price->__toString(),
    ];
  }

  /**
   * Extracts calculated price from a variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   *
   * @return array|null
   *   Calculated price data or NULL.
   */
  protected function extractCalculatedPrice($variation) {
    try {
      $store = $this->currentStore->getStore();
      if (!$store) {
        return NULL;
      }

      $context = new Context($this->currentUser, $store);

      // Get config for adjustment types if commerce_api module is enabled.
      $adjustment_types = [];
      $config = $this->configFactory->get('commerce_api.settings');
      if ($config) {
        $types = $config->get('adjustment_types');
        $adjustment_types = $types ? array_filter($types) : [];
      }

      $result = $this->priceCalculator->calculate(
        $variation,
        1,
        $context,
        $adjustment_types
      );

      $calculated_price = $result->getCalculatedPrice();
      if (!$calculated_price) {
        return NULL;
      }

      return [
        'number' => $calculated_price->getNumber(),
        'currency_code' => $calculated_price->getCurrencyCode(),
        'formatted' => $calculated_price->__toString(),
      ];
    }
    catch (\Exception $e) {
      // Price calculation failed, skip calculated price.
      return NULL;
    }
  }

  /**
   * Extracts resolved price from a variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   *
   * @return array|null
   *   Resolved price data or NULL.
   */
  protected function extractResolvedPrice($variation) {
    try {
      $store = $this->currentStore->getStore();
      if (!$store) {
        return NULL;
      }

      $context = new Context($this->currentUser, $store);
      $resolved_price = $this->chainPriceResolver->resolve(
        $variation,
        1,
        $context
      );

      if (!$resolved_price) {
        return NULL;
      }

      return [
        'number' => $resolved_price->getNumber(),
        'currency_code' => $resolved_price->getCurrencyCode(),
        'formatted' => $resolved_price->__toString(),
      ];
    }
    catch (\Exception $e) {
      // Price resolution failed, skip resolved price.
      return NULL;
    }
  }

  /**
   * Extracts and cleans product description.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   *
   * @return string
   *   Cleaned description text.
   */
  protected function extractProductDescription($product) {
    if (!$product->hasField('body') || $product->get('body')->isEmpty()) {
      return '';
    }

    $description = $product->get('body')->value;
    // Strip HTML tags for cleaner text.
    $description = strip_tags($description);
    // Limit to 1000 characters for better context.
    return substr($description, 0, 1000);
  }

  /**
   * Extracts categories from a product.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   *
   * @return array
   *   Array of category data.
   */
  protected function extractCategories($product) {
    $category_field = 'field_product_category';

    if (!$product->hasField($category_field) ||
      $product->get($category_field)->isEmpty()) {
      return [];
    }

    $categories = [];
    $entities = $product->get($category_field)->referencedEntities();
    foreach ($entities as $term) {
      $categories[] = [
        'id' => $term->id(),
        'name' => $term->label(),
      ];
    }

    return $categories;
  }

  /**
   * Gets default variation SKU and price data.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   *
   * @return array
   *   Array containing [sku, price_data].
   */
  protected function getDefaultVariationData($product) {
    $default_variation = $product->getDefaultVariation();

    if (!$default_variation ||
      $default_variation->get('status')->value != 1) {
      return ['', NULL];
    }

    $sku = $default_variation->getSku();
    $price = $this->extractVariationPrice($default_variation);

    // Remove formatted field for default price (keep it simple).
    if ($price) {
      unset($price['formatted']);
    }

    return [$sku, $price];
  }

  /**
   * Builds metadata array for a product.
   *
   * @param \Drupal\commerce_product\Entity\ProductInterface $product
   *   The product entity.
   *
   * @return array
   *   Metadata array.
   */
  protected function buildProductMetadata($product) {
    return [
      'source' => 'commerce_product',
      'product_id' => $product->id(),
      'product_type' => $product->bundle(),
      'created' => date('c', $product->getCreatedTime()),
      'changed' => date('c', $product->getChangedTime()),
      'status' => $product->get('status')->value == 1 ?
      'published' : 'unpublished',
    ];
  }

}
