<?php

namespace Drupal\commerce_ai_suite_product_recommender\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\commerce\Context;
use Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderManager;
use Drupal\commerce_order\PriceCalculatorInterface;
use Drupal\commerce_store\CurrentStoreInterface;

/**
 * Service for generating product recommendations.
 */
class RecommendationService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The AI provider manager.
   *
   * @var \Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderManager
   */
  protected $aiProviderManager;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The current store service.
   *
   * @var \Drupal\commerce_store\CurrentStoreInterface
   */
  protected $currentStore;

  /**
   * The price calculator.
   *
   * @var \Drupal\commerce_order\PriceCalculatorInterface
   */
  protected $priceCalculator;

  /**
   * Constructs a RecommendationService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderManager $ai_provider_manager
   *   The AI provider manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\commerce_store\CurrentStoreInterface $current_store
   *   The current store service.
   * @param \Drupal\commerce_order\PriceCalculatorInterface $price_calculator
   *   The price calculator.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    ConfigFactoryInterface $config_factory,
    AiProviderManager $ai_provider_manager,
    LoggerChannelFactoryInterface $logger_factory,
    AccountProxyInterface $current_user,
    ModuleHandlerInterface $module_handler,
    CurrentStoreInterface $current_store,
    PriceCalculatorInterface $price_calculator,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->aiProviderManager = $ai_provider_manager;
    $this->logger = $logger_factory
      ->get('commerce_ai_suite_product_recommender');
    $this->currentUser = $current_user;
    $this->moduleHandler = $module_handler;
    $this->currentStore = $current_store;
    $this->priceCalculator = $price_calculator;
  }

  /**
   * Gets product recommendations for the current user.
   *
   * @return array
   *   Array of recommendation data.
   */
  public function getRecommendationsForCurrentUser() {
    if (!$this->currentUser->isAuthenticated()) {
      $this->logger->info(
        'User is not authenticated, returning fallback recommendations.'
      );
      return $this->getFallbackRecommendations();
    }

    $user_id = $this->currentUser->id();
    $order_history = $this->getUserOrderHistory($user_id);

    if (empty($order_history)) {
      $this->logger->info(
        'No order history found for user @uid, ' .
        'returning fallback recommendations.',
        ['@uid' => $user_id]
      );
      return $this->getFallbackRecommendations();
    }

    $history_json = json_encode($order_history);

    // Get the configured AI provider.
    $config = $this->configFactory->get('commerce_ai_suite_product_recommender.settings');
    $provider_id = $config->get('ai_provider');

    // If no provider is configured, return fallback recommendations.
    if (empty($provider_id)) {
      $this->logger->info('No AI provider configured, returning fallback recommendations.');
      return $this->getFallbackRecommendations();
    }

    try {
      /** @var \Drupal\commerce_ai_suite_product_recommender\Plugin\AiProvider\AiProviderInterface $provider */
      $provider = $this->aiProviderManager->createInstance($provider_id);

      if (!$provider->isConfigured()) {
        $this->logger->warning(
          'AI provider @provider is not properly configured.',
          ['@provider' => $provider_id]
        );
        return $this->getFallbackRecommendations();
      }

      $recommendations = $provider->getRecommendations($history_json);

      if (empty($recommendations)) {
        $this->logger->warning(
          'No recommendations returned from @provider for user @uid.',
          ['@provider' => $provider_id, '@uid' => $user_id]
        );
        return $this->getFallbackRecommendations();
      }

      return $recommendations;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to get recommendations from @provider: @message',
        ['@provider' => $provider_id, '@message' => $e->getMessage()]
      );
      return $this->getFallbackRecommendations();
    }
  }

  /**
   * Gets last 5 recent order history for a given user.
   *
   * @param int $user_id
   *   The user ID.
   *
   * @return array
   *   Array of order history data.
   */
  protected function getUserOrderHistory($user_id) {
    try {
      $order_storage = $this->entityTypeManager
        ->getStorage('commerce_order');

      // Load completed orders for the user.
      $query = $order_storage->getQuery()
        ->condition('uid', $user_id)
        ->condition('state', 'completed')
        ->sort('completed', 'DESC')
        ->range(0, 5)
        ->accessCheck(FALSE);

      $order_ids = $query->execute();

      if (empty($order_ids)) {
        return [];
      }

      $orders = $order_storage->loadMultiple($order_ids);
      $history = [];

      foreach ($orders as $order) {
        foreach ($order->getItems() as $order_item) {
          $purchased_entity = $order_item->getPurchasedEntity();
          if (!$purchased_entity) {
            continue;
          }

          $product = $purchased_entity->getProduct();
          if (!$product) {
            continue;
          }

          $categories = [];
          if ($product->hasField('field_product_category') &&
            !$product->get('field_product_category')->isEmpty()) {
            $field_categories = $product->get('field_product_category')
              ->referencedEntities();
            foreach ($field_categories as $term) {
              $categories[] = $term->label();
            }
          }

          $history[] = [
            'product_id' => $product->id(),
            'product_title' => $product->label(),
            'sku' => $purchased_entity->getSku(),
            'quantity' => (int) $order_item->getQuantity(),
            'categories' => $categories,
            'order_date' => date('c', $order->getCompletedTime()),
          ];
        }
      }

      return $history;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to get order history for user @uid: @message',
        [
          '@uid' => $user_id,
          '@message' => $e->getMessage(),
        ]
      );
      return [];
    }
  }

  /**
   * Enriches recommendations with full product entity data.
   *
   * @param array $recommendations
   *   Array of recommendation data with at least 'id' key.
   *
   * @return array
   *   Array of enriched recommendation data with product details.
   */
  public function enrichRecommendationsWithProductData(array $recommendations) {
    if (empty($recommendations)) {
      return [];
    }

    // Load actual product entities.
    $product_ids = array_column($recommendations, 'id');
    $product_storage = $this->entityTypeManager
      ->getStorage('commerce_product');
    $products = $product_storage->loadMultiple($product_ids);

    // Build enriched data for each recommendation.
    $product_data = [];
    foreach ($recommendations as $recommendation) {
      $product_id = $recommendation['id'];
      if (isset($products[$product_id])) {
        $product = $products[$product_id];

        // Get product URL.
        $url = $product->toUrl()->toString();

        // Get product image.
        $image_url = NULL;
        if ($product->hasField('field_images') &&
          !$product->get('field_images')->isEmpty()) {
          $media_field = $product->get('field_images')->first();
          if ($media_field && $media_field->entity) {
            /** @var \Drupal\media\Entity\Media $media */
            $media = $media_field->entity;

            // Get source field from the media entity.
            $source_field = $media->getSource()
              ->getSourceFieldDefinition($media->bundle->entity)
              ->getName();

            if ($media->hasField($source_field) &&
              !$media->get($source_field)->isEmpty()) {
              $file = $media->get($source_field)->first();
              if ($file && $file->entity) {
                $image_url = $file->entity->createFileUrl();
              }
            }
          }
        }

        // Get price and currency from default variation.
        $price_number = NULL;
        $currency_code = NULL;
        $default_variation = $product->getDefaultVariation();
        if ($default_variation && $default_variation->get('status')->value == 1) {
          // Try to get calculated price first.
          $calculated_price = $this->getCalculatedPrice($default_variation);
          if ($calculated_price) {
            $price_number = number_format(
              (float) $calculated_price->getNumber(),
              2,
              '.',
              ''
            );
            $currency_code = $calculated_price->getCurrencyCode();
          }
          // Fallback to base price if calculated price not available.
          elseif ($default_variation->hasField('price')) {
            $price_field = $default_variation->get('price')->first();
            if ($price_field && method_exists($price_field, 'toPrice')) {
              $price = $price_field->toPrice();
              $price_number = number_format(
                (float) $price->getNumber(),
                2,
                '.',
                ''
              );
              $currency_code = $price->getCurrencyCode();
            }
          }
        }

        $product_data[] = [
          'id' => $product_id,
          'sku' => $default_variation ? $default_variation->getSku() : NULL,
          'title' => $product->label(),
          'url' => $url,
          'image_url' => $image_url,
          'price' => $price_number,
          'currency_code' => $currency_code,
          'reason' => $recommendation['reason'] ?? NULL,
          'product' => $product,
        ];
      }
    }

    // Allow other modules to alter the recommendation data.
    $this->moduleHandler->alter(
      'commerce_ai_suite_product_recommender_recommendation_data',
      $product_data
    );
    // Unset the product entity before returning.
    foreach ($product_data as &$item) {
      unset($item['product']);
    }

    return $product_data;
  }

  /**
   * Gets calculated price for a product variation.
   *
   * @param \Drupal\commerce_product\Entity\ProductVariationInterface $variation
   *   The product variation.
   *
   * @return \Drupal\commerce_price\Price|null
   *   The calculated price or NULL if calculation fails.
   */
  protected function getCalculatedPrice($variation) {
    try {
      $store = $this->currentStore->getStore();
      if (!$store) {
        return NULL;
      }

      $context = new Context($this->currentUser, $store);

      // Get config for adjustment types if commerce_api module is enabled.
      $adjustment_types = [];
      $config = $this->configFactory->get('commerce_api.settings');
      if ($config) {
        $types = $config->get('adjustment_types');
        $adjustment_types = $types ? array_filter($types) : [];
      }

      $result = $this->priceCalculator->calculate(
        $variation,
        1,
        $context,
        $adjustment_types
      );

      return $result->getCalculatedPrice();
    }
    catch (\Exception $e) {
      // Price calculation failed, return NULL.
      return NULL;
    }
  }

  /**
   * Gets fallback recommendations (top sellers or featured products).
   *
   * @return array
   *   Array of fallback recommendations.
   */
  protected function getFallbackRecommendations() {
    try {
      $product_storage = $this->entityTypeManager
        ->getStorage('commerce_product');

      // Query for featured or best-selling products.
      $query = $product_storage->getQuery()
        ->condition('status', 1)
        ->range(0, 5)
        ->accessCheck(FALSE);

      // If you have a field for featured products, add it here.
      // $query->condition('field_featured', 1);.
      $product_ids = $query->execute();

      if (empty($product_ids)) {
        return [];
      }

      $products = $product_storage->loadMultiple($product_ids);
      $recommendations = [];

      foreach ($products as $product) {
        $default_variation = $product->getDefaultVariation();
        $price = NULL;
        $currency_code = NULL;
        if ($default_variation &&
          $default_variation->hasField('price')) {
          $price_field = $default_variation->get('price')->first();
          if ($price_field) {
            $price_obj = $price_field->toPrice();
            // Ensure price is formatted as a number with two decimal places.
            $price = number_format((float) $price_obj->getNumber(), 2, '.', '');
            $currency_code = $price_obj->getCurrencyCode();
          }
        }

        $recommendations[] = [
          'id' => $product->id(),
          'title' => $product->label(),
          'price' => $price,
          'currency_code' => $currency_code,
          'reason' => 'Top seller',
        ];
      }

      return $recommendations;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to get fallback recommendations: @message',
        ['@message' => $e->getMessage()]
      );
      return [];
    }
  }

}
