<?php

namespace Drupal\commerce_ai_suite_product_recommender\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_ai_suite\Service\KeyHelper;
use Google\Auth\Credentials\ServiceAccountCredentials;
use Google\Auth\Middleware\AuthTokenMiddleware;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\HandlerStack;

/**
 * Service for Discovery Engine Agentspace with google/auth SA handling.
 */
class VertexAiClientService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The key helper service.
   *
   * @var \Drupal\commerce_ai_suite\Service\KeyHelper
   */
  protected $keyHelper;

  /**
   * Cached middleware.
   *
   * @var \Google\Auth\Middleware\AuthTokenMiddleware|null
   */
  protected $middleware;

  /**
   * Scopes for creds.
   *
   * @var array
   */
  protected $scopes = ['https://www.googleapis.com/auth/cloud-platform'];

  /**
   * Constructs a VertexAiClientService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\commerce_ai_suite\Service\KeyHelper $key_helper
   *   The key helper service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    KeyHelper $key_helper,
  ) {
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('commerce_ai_suite_product_recommender');
    $this->keyHelper = $key_helper;
  }

  /**
   * Gets middleware for Guzzle using SA creds.
   *
   * @return \Google\Auth\Middleware\AuthTokenMiddleware|null
   *   The auth middleware or NULL on failure.
   */
  protected function getAuthMiddleware() {
    if ($this->middleware) {
      return $this->middleware;
    }

    $config_name = 'commerce_ai_suite_product_recommender.settings';
    $config = $this->configFactory->get($config_name);
    $key_id = $config->get('gcs_service_account_key');

    if (empty($key_id)) {
      $this->logger->error('GCS service account key is not configured.');
      return NULL;
    }

    try {
      // Load the key using the helper service.
      $sa_json = $this->keyHelper->loadServiceAccountKey($key_id);
    }
    catch (\Exception $e) {
      $this->logger->error($e->getMessage());
      return NULL;
    }

    try {
      // Create ServiceAccountCredentials with scopes and JSON.
      $creds = new ServiceAccountCredentials($this->scopes, $sa_json);

      // Middleware for Guzzle.
      $this->middleware = new AuthTokenMiddleware($creds);
      $this->logger->info('Auth middleware initialized from SA.');
      return $this->middleware;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Middleware creation failed: @msg',
        ['@msg' => $e->getMessage()]
      );
      return NULL;
    }
  }

  /**
   * Queries Discovery Engine for recommendations.
   *
   * @param string $user_history
   *   JSON-encoded history.
   *
   * @return array|null
   *   Array of recommendations or NULL on failure.
   */
  public function getRecommendations($user_history) {
    $middleware = $this->getAuthMiddleware();
    if (!$middleware) {
      return NULL;
    }

    $config = $this->loadConfiguration();
    if (!$config) {
      return NULL;
    }

    $prompt = $this->preparePrompt($user_history, $config['agent_prompt']);
    $urls = $this->buildApiUrls(
      $config['gcp_project_id'],
      $config['agentspace_app_id']
    );

    try {
      $client = $this->createAuthenticatedClient($middleware);
      $request_body = $this->buildRequestBody($prompt, $urls['session']);
      $response = $client->post($urls['answer'], [
        'json' => $request_body,
        'timeout' => 30,
      ]);

      return $this->processResponse($response);
    }
    catch (GuzzleException $e) {
      $this->logger->error('Guzzle error: @msg', ['@msg' => $e->getMessage()]);
      return NULL;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Unexpected error: @msg',
        ['@msg' => $e->getMessage()]
      );
      return NULL;
    }
  }

  /**
   * Loads and validates configuration.
   *
   * @return array|null
   *   Configuration array or NULL if validation fails.
   */
  protected function loadConfiguration() {
    $config_name = 'commerce_ai_suite_product_recommender.settings';
    $config = $this->configFactory->get($config_name);

    $gcp_project_id = $config->get('gcp_project_id') ?: '';
    $agentspace_app_id = $config->get('agentspace_app_id') ?: '';
    $agent_prompt = $config->get('agent_prompt') ?: '';

    if (empty($gcp_project_id) ||
      empty($agentspace_app_id) ||
      empty($agent_prompt)) {
      $this->logger->error(
        'GCP Project ID, Agentspace App ID, or prompt missing.'
      );
      return NULL;
    }

    return [
      'gcp_project_id' => $gcp_project_id,
      'agentspace_app_id' => $agentspace_app_id,
      'agent_prompt' => $agent_prompt,
    ];
  }

  /**
   * Prepares the prompt with user history.
   *
   * @param string $user_history
   *   JSON-encoded user history.
   * @param string $agent_prompt
   *   The base prompt template.
   *
   * @return string
   *   The prepared prompt.
   */
  protected function preparePrompt($user_history, $agent_prompt) {
    $history_summary = $this->summarizeHistory($user_history);
    $prompt = str_replace('[HISTORY_JSON]', $history_summary, $agent_prompt);
    // Cap at 2000 chars for safety.
    return trim(substr($prompt, 0, 2000));
  }

  /**
   * Builds API URLs for Discovery Engine.
   *
   * @param string $gcp_project_id
   *   GCP project ID.
   * @param string $agentspace_app_id
   *   Agentspace application ID.
   *
   * @return array
   *   Array with 'answer' and 'session' URL paths.
   */
  protected function buildApiUrls($gcp_project_id, $agentspace_app_id) {
    $location = 'global';
    $collection = 'default_collection';
    $engine_path = "projects/{$gcp_project_id}/locations/{$location}/" .
      "collections/{$collection}/engines/{$agentspace_app_id}";

    return [
      'answer' => "https://discoveryengine.googleapis.com/v1alpha/" .
      "{$engine_path}/servingConfigs/default_search:answer",
      'session' => "{$engine_path}/sessions/-",
    ];
  }

  /**
   * Creates an authenticated Guzzle client.
   *
   * @param \Google\Auth\Middleware\AuthTokenMiddleware $middleware
   *   The auth middleware.
   *
   * @return \GuzzleHttp\Client
   *   The authenticated client.
   */
  protected function createAuthenticatedClient($middleware) {
    $stack = HandlerStack::create();
    $stack->push($middleware);

    return new Client([
      'handler' => $stack,
      'auth' => 'google_auth',
    ]);
  }

  /**
   * Builds the request body for Discovery Engine.
   *
   * @param string $prompt
   *   The prepared prompt.
   * @param string $session_path
   *   The session path.
   *
   * @return array
   *   The request body.
   */
  protected function buildRequestBody($prompt, $session_path) {
    return [
      'query' => ['text' => $prompt],
      'session' => $session_path,
      'userPseudoId' => 'anonymous-user',
      'answerGenerationSpec' => [
        'ignoreLowRelevantContent' => TRUE,
        'includeCitations' => TRUE,
        'modelSpec' => ['modelVersion' => 'stable'],
        'promptSpec' => ['preamble' => $prompt],
      ],
      'groundingSpec' => ['includeGroundingSupports' => TRUE],
      'searchSpec' => [
        'searchParams' => [
          'maxReturnResults' => 5,
        ],
      ],
    ];
  }

  /**
   * Processes the API response.
   *
   * @param \Psr\Http\Message\ResponseInterface $response
   *   The HTTP response.
   *
   * @return array
   *   Array of recommendations or empty array.
   */
  protected function processResponse($response) {
    if ($response->getStatusCode() !== 200) {
      $this->logger->error('Answer failed: @code - @body', [
        '@code' => $response->getStatusCode(),
        '@body' => $response->getBody()->getContents(),
      ]);
      return NULL;
    }

    $body = json_decode($response->getBody()->getContents(), TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      $this->logger->error(
        'Parse error: @error',
        ['@error' => json_last_error_msg()]
      );
      return NULL;
    }

    $answer = $body['answer'] ?? NULL;
    if (empty($answer)) {
      $this->logger->warning('No answer generated.');
      return [];
    }

    $answer_text = $answer['answerText'] ?? '';
    if (empty($answer_text)) {
      $this->logger->warning('Empty answer text.');
      return [];
    }

    return $this->extractRecommendations($answer_text);
  }

  /**
   * Extracts recommendations from answer text.
   *
   * @param string $answer_text
   *   The answer text from the API.
   *
   * @return array
   *   Array of recommendations or empty array.
   */
  protected function extractRecommendations($answer_text) {
    // Extract JSON from markdown: ```json.
    $json_str = $answer_text;
    if (preg_match('/```json\s*(.*?)\s*```/s', $answer_text, $matches)) {
      $json_str = trim($matches[1]);
      $this->logger->info('Extracted JSON from markdown wrapper.');
    }

    $recommendations_data = json_decode($json_str, TRUE);
    if (json_last_error() !== JSON_ERROR_NONE ||
      !isset($recommendations_data['recommendations'])) {
      $this->logger->warning(
        'Invalid JSON in answerText: @error. Raw: @raw',
        [
          '@error' => json_last_error_msg(),
          '@raw' => substr($json_str, 0, 200),
        ]
      );
      return [];
    }

    return $recommendations_data['recommendations'];
  }

  /**
   * Summarizes user order history to key insights for prompt injection.
   *
   * @param string $raw_history_json
   *   Raw JSON array of orders.
   *   e.g., [{
   *    "sku": "123",
   *    "categories": "Shoes",
   *    "order_date": "2025-10-01"}
   *   ]).
   *
   * @return string
   *   Concise summary (~200-400 chars) of preferences/trends.
   */
  protected function summarizeHistory($raw_history_json) {
    $history = json_decode($raw_history_json, TRUE);
    if (json_last_error() !== JSON_ERROR_NONE || empty($history)) {
      $this->logger->warning('Invalid history JSON; using empty summary.');
      // Fallback to prompt's sparse handling.
      return '{}';
    }

    // Aggregate: Top categories (count), recent SKUs (last 10), total value
    // proxy.
    // Flatten all categories from all history items.
    $all_categories = [];
    foreach ($history as $item) {
      if (!empty($item['categories']) && is_array($item['categories'])) {
        foreach ($item['categories'] as $category) {
          $all_categories[] = $category;
        }
      }
    }

    // Count category occurrences and get top 3.
    $cats = !empty($all_categories) ?
      array_count_values($all_categories) : [];
    arsort($cats);
    $top_cats = array_slice($cats, 0, 3, TRUE);

    // Last 10 purchases.
    $recent_skus = array_slice(array_column($history, 'sku'), -10);

    $summary = sprintf(
    'Preferences: %s. Recent purchase SKUs: %s. Focus on upsells/complements.',
    json_encode(array_keys($top_cats)),
    implode(', ', $recent_skus),
    );

    $this->logger->info('History summarized: @chars chars', ['@chars' => strlen($summary)]);
    return $summary;
  }

}
