<?php

namespace Drupal\commerce_ai_suite\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Google\Cloud\Storage\StorageClient;

/**
 * Shared service for interacting with Google Cloud Storage.
 *
 * This service provides reusable GCS functionality for all Commerce AI
 * submodules.
 */
class GcsClientService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The key helper service.
   *
   * @var \Drupal\commerce_ai_suite\Service\KeyHelper
   */
  protected $keyHelper;

  /**
   * Cache of storage clients keyed by config name.
   *
   * @var array
   */
  protected $storageClients = [];

  /**
   * Constructs a GcsClientService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\commerce_ai_suite\Service\KeyHelper $key_helper
   *   The key helper service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    KeyHelper $key_helper,
  ) {
    $this->configFactory = $config_factory;
    $this->logger = $logger_factory->get('commerce_ai_suite');
    $this->keyHelper = $key_helper;
  }

  /**
   * Gets the GCS storage client for a given configuration.
   *
   * @param string $config_name
   *   The configuration object name.
   *   E.g., 'commerce_ai_suite_product_recommender.settings'.
   *
   * @return \Google\Cloud\Storage\StorageClient|null
   *   The storage client or NULL if initialization fails.
   */
  public function getStorageClient($config_name) {
    if (isset($this->storageClients[$config_name])) {
      return $this->storageClients[$config_name];
    }

    try {
      $config = $this->configFactory->get($config_name);
      $key_id = $config->get('gcs_service_account_key');

      if (empty($key_id)) {
        throw new \Exception(
          'GCS service account key is not configured.'
        );
      }

      // Load the key using the helper service.
      $key_data = $this->keyHelper->loadServiceAccountKey($key_id);

      $this->storageClients[$config_name] = new StorageClient([
        'keyFile' => $key_data,
      ]);

      return $this->storageClients[$config_name];
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to initialize GCS client for @config: @message',
        [
          '@config' => $config_name,
          '@message' => $e->getMessage(),
        ]
      );
      return NULL;
    }
  }

  /**
   * Uploads a file to GCS.
   *
   * @param string $config_name
   *   The configuration object name.
   * @param string $bucket_name
   *   The GCS bucket name.
   * @param string $file_content
   *   The content to upload.
   * @param string $destination_path
   *   The destination path in the bucket.
   *   E.g., "products/chunk_1.jsonl".
   * @param string $content_type
   *   The content type (default: application/jsonl).
   *
   * @return bool
   *   TRUE if upload succeeded, FALSE otherwise.
   */
  public function uploadFile(
    $config_name,
    $bucket_name,
    $file_content,
    $destination_path,
    $content_type = 'application/jsonl',
  ) {
    $client = $this->getStorageClient($config_name);
    if (!$client) {
      return FALSE;
    }

    try {
      if (empty($bucket_name)) {
        throw new \Exception('GCS bucket name is not configured.');
      }

      $bucket = $client->bucket($bucket_name);
      $bucket->upload($file_content, [
        'name' => $destination_path,
        'metadata' => [
          'contentType' => $content_type,
        ],
      ]);

      $this->logger->info(
        'Uploaded file to GCS: @path',
        ['@path' => $destination_path]
      );
      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to upload file to GCS: @message',
        ['@message' => $e->getMessage()]
      );
      return FALSE;
    }
  }

  /**
   * Uploads a file with retry logic.
   *
   * @param string $config_name
   *   The configuration object name.
   * @param string $bucket_name
   *   The GCS bucket name.
   * @param string $file_content
   *   The content to upload.
   * @param string $destination_path
   *   The destination path in the bucket.
   * @param string $content_type
   *   The content type.
   * @param int $max_retries
   *   Maximum number of retry attempts (default: 3).
   *
   * @return bool
   *   TRUE if upload succeeded, FALSE otherwise.
   */
  public function uploadFileWithRetry(
    $config_name,
    $bucket_name,
    $file_content,
    $destination_path,
    $content_type = 'application/jsonl',
    $max_retries = 3,
  ) {
    $attempt = 0;

    while ($attempt < $max_retries) {
      $result = $this->uploadFile(
        $config_name,
        $bucket_name,
        $file_content,
        $destination_path,
        $content_type
      );
      if ($result) {
        return TRUE;
      }

      $attempt++;
      if ($attempt < $max_retries) {
        $delay = 2 * pow(2, $attempt - 1);
        $this->logger->warning(
          'Upload failed, retry in @delay sec (Attempt @attempt/@max)',
          [
            '@delay' => $delay,
            '@attempt' => $attempt,
            '@max' => $max_retries,
          ]
        );
        sleep($delay);
      }
    }

    $this->logger->error(
      'Failed to upload file after @max attempts: @path',
      [
        '@max' => $max_retries,
        '@path' => $destination_path,
      ]
    );
    return FALSE;
  }

  /**
   * Lists files in a GCS bucket with a given prefix.
   *
   * @param string $config_name
   *   The configuration object name.
   * @param string $bucket_name
   *   The GCS bucket name.
   * @param string $prefix
   *   The prefix to filter files (e.g., "products/").
   *
   * @return array
   *   An array of file names.
   */
  public function listFiles($config_name, $bucket_name, $prefix = '') {
    $client = $this->getStorageClient($config_name);
    if (!$client) {
      return [];
    }

    try {
      $bucket = $client->bucket($bucket_name);
      $objects = $bucket->objects(['prefix' => $prefix]);
      $files = [];

      foreach ($objects as $object) {
        $files[] = $object->name();
      }

      return $files;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to list files from GCS: @message',
        ['@message' => $e->getMessage()]
      );
      return [];
    }
  }

  /**
   * Deletes a file from GCS.
   *
   * @param string $config_name
   *   The configuration object name.
   * @param string $bucket_name
   *   The GCS bucket name.
   * @param string $file_path
   *   The file path in the bucket.
   *
   * @return bool
   *   TRUE if deletion succeeded, FALSE otherwise.
   */
  public function deleteFile($config_name, $bucket_name, $file_path) {
    $client = $this->getStorageClient($config_name);
    if (!$client) {
      return FALSE;
    }

    try {
      $bucket = $client->bucket($bucket_name);
      $object = $bucket->object($file_path);
      $object->delete();

      $this->logger->info(
        'Deleted file from GCS: @path',
        ['@path' => $file_path]
      );
      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error(
        'Failed to delete file from GCS: @message',
        ['@message' => $e->getMessage()]
      );
      return FALSE;
    }
  }

}
