<?php

namespace Drupal\commerce_ai_suite\Service;

use Drupal\key\KeyRepositoryInterface;

/**
 * Helper service for loading and validating keys from the Key module.
 */
class KeyHelper {

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * Constructs a KeyHelper object.
   *
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository.
   */
  public function __construct(KeyRepositoryInterface $key_repository) {
    $this->keyRepository = $key_repository;
  }

  /**
   * Loads and decodes a service account key.
   *
   * @param string $key_id
   *   The key ID.
   *
   * @return array|null
   *   The decoded service account key data or NULL on failure.
   *
   * @throws \Exception
   *   Throws exception if key is invalid.
   */
  public function loadServiceAccountKey($key_id) {
    if (empty($key_id)) {
      throw new \Exception('Key ID is empty.');
    }

    $key = $this->keyRepository->getKey($key_id);

    if (!$key) {
      throw new \Exception(
        sprintf('Key "%s" could not be loaded.', $key_id)
      );
    }

    $key_value = $key->getKeyValue();
    if (empty($key_value)) {
      throw new \Exception(
        sprintf('Key "%s" is empty.', $key_id)
      );
    }

    $key_data = json_decode($key_value, TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      throw new \Exception(
        sprintf(
          'Key "%s" does not contain valid JSON: %s',
          $key_id,
          json_last_error_msg()
        )
      );
    }

    return $key_data;
  }

  /**
   * Validates that a key contains a service account.
   *
   * @param array $key_data
   *   The decoded key data.
   *
   * @return bool
   *   TRUE if valid service account, FALSE otherwise.
   */
  public function validateServiceAccountKey(array $key_data) {
    return !empty($key_data['type']) &&
      $key_data['type'] === 'service_account';
  }

}
