<?php

declare(strict_types=1);

namespace Drupal\commerce_alma_test\Api\Endpoints;

use Alma\API\Endpoints\Results\Eligibility;
use Alma\API\Entities\Payment;
use Alma\API\RequestError;
use Drupal\Core\Url;

/**
 * Provides a test payment class for the Commerce Alma module.
 */
class Payments {

  /**
   * The valid payment ID for testing.
   *
   * @var string
   */
  const VALID_PID = '1234567890';

  /**
   * Fetches a payment by its ID.
   *
   * @param string $pid
   *   The payment ID to fetch.
   *
   * @return \Alma\API\Entities\Payment
   *   The payment entity.
   *
   * @throws \Alma\API\RequestError
   *   If the payment ID is invalid.
   */
  public function fetch(string $pid) {
    if ($pid !== self::VALID_PID) {
      throw new RequestError();
    }

    return new Payment([
      'id' => self::VALID_PID,
      'purchase_amount' => 2990,
      'state' => Payment::STATE_PAID,
    ]);
  }

  /**
   * Flags a payment as potential fraud.
   *
   * @param string $id
   *   The payment ID to flag.
   * @param string|null $reason
   *   The reason for flagging the payment, if any.
   *
   * @return bool
   *   TRUE if the operation was successful, FALSE otherwise.
   */
  public function flagAsPotentialFraud($id, $reason = NULL): bool {
    return TRUE;
  }

  /**
   * Checks the eligibility of a payment.
   *
   * @param array $data
   *   The data to check eligibility against.
   * @param bool $raiseOnError
   *   Whether to raise an error on failure.
   *
   * @return \Alma\API\Endpoints\Results\Eligibility[]
   *   An array of eligibility results.
   */
  public function eligibility(array $data, $raiseOnError = FALSE): array {
    return [new Eligibility(['eligible' => TRUE])];
  }

  /**
   * Creates a new payment instance.
   *
   * @param array $data
   *   The data to create the payment with.
   *
   * @return \Alma\API\Entities\Payment
   *   The created payment entity.
   */
  public function create($data): Payment {
    return new Payment([
      'id' => self::VALID_PID,
      'purchase_amount' => 2990,
      'state' => Payment::STATE_PAID,
      'url' => Url::fromRoute(
        'commerce_alma_test.notify',
        [],
        ['absolute' => TRUE]
      )->toString(),
    ]);
  }

  /**
   * Totally refund a payment.
   *
   * @param string $id
   *   ID of the payment to be refunded.
   * @param string $merchantReference
   *   Merchant reference for the refund to be executed.
   * @param string $comment
   *   Comment for the refund to be executed.
   *
   * @return \Alma\API\Entities\Payment
   *   The payment entity after the refund.
   *
   * @throws \Alma\API\RequestError
   *   If the payment ID is invalid or the refund fails.
   */
  public function fullRefund($id, $merchantReference = "", $comment = "") {
    if ($id !== self::VALID_PID) {
      throw new RequestError();
    }

    return new Payment([
      'id' => self::VALID_PID,
      'purchase_amount' => 2990,
      'state' => Payment::STATE_PAID,
    ]);
  }

  /**
   * Refund a payment partially.
   *
   * @param string $id
   *   ID of the payment to be refunded.
   * @param int $amount
   *   Amount that should be refunded.
   *   Must be expressed as a cents integer.
   * @param string $merchantReference
   *   Merchant reference for the refund to be executed.
   * @param string $comment
   *   Comment for the refund to be executed.
   *
   * @return \Alma\API\Entities\Payment
   *   The payment entity after the partiel refund.
   *
   * @throws \Alma\API\RequestError
   *   If the payment ID is invalid or the refund fails.
   */
  public function partialRefund($id, $amount, $merchantReference = "", $comment = "") {
    if ($id !== self::VALID_PID) {
      throw new RequestError();
    }

    return new Payment([
      'id' => self::VALID_PID,
      'purchase_amount' => 2990,
      'state' => Payment::STATE_PAID,
    ]);
  }

}
