<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_alma\Kernel;

use Alma\API\Entities\Payment as RemotePayment;
use Drupal\commerce_alma_test\Api\Endpoints\Payments;
use Drupal\commerce_price\Price;

/**
 * Tests the payment update.
 */
class CronTest extends KernelTestBase {

  /**
   * The order storage.
   *
   * @var \Drupal\commerce_order\OrderStorage
   */
  protected $orderStorage;

  /**
   * The Alma payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $almaGateway;

  /**
   * A non Alma payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $altGateway;

  /**
   * The Alma payment gateway with no payments update.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $almaGatewayNoPaymentsUpdate;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->orderStorage = $this->container->get('entity_type.manager')->getStorage('commerce_order');

    $this->almaGateway = $this->createGateway([
      'id' => 'offsite',
      'label' => $this->randomString(),
      'plugin' => 'alma',
      'configuration' => [
        'payment_method_types' => ['credit_card'],
        'api_key' => static::TEST_API_KEY,
        'mode' => 'live',
        'fee_plan' => 'general_4_0_0',
      ],
    ]);

    $this->almaGatewayNoPaymentsUpdate = $this->createGateway([
      'id' => 'offsite_no_update',
      'label' => $this->randomString(),
      'plugin' => 'alma',
      'configuration' => [
        'payment_method_types' => ['credit_card'],
        'api_key' => static::TEST_API_KEY,
        'mode' => 'live',
        'fee_plan' => 'general_4_0_0',
        'update_payments' => FALSE,
      ],
    ]);

    $this->altGateway = $this->createGateway([
      'id' => 'alt',
      'label' => $this->randomString(),
      'plugin' => 'manual',
    ]);
  }

  /**
   * Tests update (cron and queue worker).
   *
   * @dataProvider testRunDataProvider
   */
  public function testRun(array $payment_values): void {
    $this->createPayment($payment_values);
    $this->container->get('commerce_alma.cron')->run();
    $queue = \Drupal::queue('commerce_alma_payment_updater');
    $this->assertEquals(1, $queue->numberOfItems());
  }

  /**
   * Data provider for testRunAbort.
   *
   * @return array
   *   The data provider.
   */
  public static function testRunDataProvider(): array {
    $data = [];
    $data[] = [
      [
        'type' => 'payment_default',
        'payment_gateway' => 'offsite',
        'remote_id' => Payments::VALID_PID,
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
        'remote_state' => RemotePayment::STATE_IN_PROGRESS,
      ],
    ];
    return $data;
  }

  /**
   * Tests update (cron and queue worker).
   *
   * @dataProvider testRunAbortDataProvider
   */
  public function testRunAbort(array $payment_values): void {
    $this->createPayment($payment_values);
    $this->container->get('commerce_alma.cron')->run();
    $queue = \Drupal::queue('commerce_alma_payment_updater');
    $this->assertEquals(0, $queue->numberOfItems());
  }

  /**
   * Data provider for testRunAbort.
   *
   * @return array
   *   The data provider.
   */
  public static function testRunAbortDataProvider(): array {
    $data = [];
    // The remote state is invalid.
    $data[] = [
      [
        'type' => 'payment_default',
        'payment_gateway' => 'offsite',
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
        'remote_state' => RemotePayment::STATE_PAID,
      ],
    ];
    // The gateway configuration do not allow payments update.
    $data[] = [
      [
        'type' => 'payment_default',
        'payment_gateway' => 'offsite_no_update',
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
        'remote_state' => RemotePayment::STATE_IN_PROGRESS,
      ],
    ];
    // The gateway is not valid.
    $data[] = [
      [
        'type' => 'payment_default',
        'payment_gateway' => 'alt',
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
        'remote_state' => RemotePayment::STATE_IN_PROGRESS,
      ],
    ];
    return $data;
  }

}
