<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_alma\Kernel;

use Drupal\commerce_price\Price;

/**
 * Tests the FilterPaymentGatewaysEvent.
 *
 * @group commerce
 */
class FilterPaymentGatewaysEventSubscriverTest extends KernelTestBase {

  /**
   * The payment gateway storage.
   *
   * @var \Drupal\commerce_payment\PaymentGatewayStorageInterface
   */
  protected $storage;

  /**
   * The payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $almaGateway;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->storage = $this->container->get('entity_type.manager')->getStorage('commerce_payment_gateway');

    $this->almaGateway = $this->createGateway([
      'id' => 'offsite',
      'label' => $this->randomString(),
      'plugin' => 'alma',
      'configuration' => [
        'payment_method_types' => ['credit_card'],
        'api_key' => static::TEST_API_KEY,
        'mode' => 'live',
        'fee_plan' => 'general_4_0_0',
      ],
    ]);
  }

  /**
   * Tests that the proper gateway is filtered out.
   *
   * @dataProvider testAvailableGatewayDataProvider
   */
  public function testAvailableGateway(array $order_values): void {
    $order = $this->createOrder($order_values);
    $available_payment_gateways = $this->storage->loadMultipleForOrder($order);
    $this->assertEquals(1, count($available_payment_gateways));
  }

  /**
   * Data provider for testEvent.
   *
   * @return array
   *   The data provider.
   */
  public static function testAvailableGatewayDataProvider(): array {
    $data = [];
    $data[] = [
      [
        'type' => 'default',
        'store_id' => 1,
        'items' => [
          [
            'type' => 'default',
            'quantity' => 1,
            'unit_price' => new Price('29.90', 'EUR'),
          ],
        ],
      ],
    ];
    return $data;
  }

  /**
   * Tests that the proper gateway is filtered out.
   *
   * @dataProvider testUnvailableGatewayDataProvider
   */
  public function testUnvailableGateway(array $order_values): void {
    $order = $this->createOrder($order_values);
    $available_payment_gateways = $this->storage->loadMultipleForOrder($order);
    $this->assertEquals(0, count($available_payment_gateways));
  }

  /**
   * Data provider for testEvent.
   *
   * @return array
   *   The data provider.
   */
  public static function testUnvailableGatewayDataProvider(): array {
    $data = [];
    // No items in the order = no total price.
    $data[] = [
      [
        'type' => 'default',
        'store_id' => 1,
        'items' => [],
      ],
    ];
    // Total prices is zero.
    $data[] = [
      [
        'type' => 'default',
        'store_id' => 1,
        'items' => [
          [
            'type' => 'default',
            'quantity' => 1,
            'unit_price' => new Price('0', 'EUR'),
          ],
        ],
      ],
    ];
    // Currency is not EUR.
    $data[] = [
      [
        'type' => 'default',
        'store_id' => 1,
        'items' => [
          [
            'type' => 'default',
            'quantity' => 1,
            'unit_price' => new Price('29.90', 'USD'),
          ],
        ],
      ],
    ];
    return $data;
  }

}
