<?php

namespace Drupal\Tests\commerce_alma\Kernel;

use Alma\API\Entities\Payment as RemotePayment;
use Drupal\commerce_alma_test\Api\Endpoints\Payments;
use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_price\Price;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Alma kernel test.
 */
class OffsitePaymentGatewayTest extends KernelTestBase {

  /**
   * The payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $almaGateway;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->almaGateway = $this->createGateway([
      'id' => 'offsite',
      'label' => $this->randomString(),
      'plugin' => 'alma',
      'configuration' => [
        'payment_method_types' => ['credit_card'],
        'api_key' => static::TEST_API_KEY,
        'mode' => 'live',
        'fee_plan' => 'general_4_0_0',
      ],
    ]);
  }

  /**
   * Gets valid payment values for testing.
   *
   * @return array
   *   The valid payment values.
   */
  protected function getValidPaymentValues(): array {
    return [
      'type' => 'payment_default',
      'payment_gateway' => 'offsite',
      'remote_id' => Payments::VALID_PID,
      'amount' => new Price('29.90', 'EUR'),
      'order_id' => 1,
    ];
  }

  /**
   * Gets valid order values for testing.
   *
   * @return array
   *   The valid order values.
   */
  protected function getValidOrderValues() {
    return [
      'type' => 'default',
      'store_id' => 1,
      'items' => [
        [
          'type' => 'default',
          'quantity' => 1,
          'unit_price' => new Price('29.90', 'EUR'),
        ],
      ],
    ];
  }

  /**
   * Gets a valid request for testing.
   *
   * @return \Symfony\Component\HttpFoundation\Request
   *   The valid request.
   */
  protected function getValidRequest() {
    return new Request(['pid' => Payments::VALID_PID]);
  }

  /**
   * Tests the payment gateway plugin onReturn() method.
   */
  public function testOnReturn() {
    $order = $this->createOrder($this->getValidOrderValues());
    $this->createPayment($this->getValidPaymentValues());
    $plugin = $this->almaGateway->getPlugin();

    $this->assertNull($plugin->onReturn($order, $this->getValidRequest()));
  }

  /**
   * Tests the payment gateway plugin onReturn() method.
   *
   * @dataProvider testOnReturnExceptionDataProvider
   */
  public function testOnReturnException(array $order_values, Request $request) {
    $order = Order::create($order_values);
    $plugin = $this->almaGateway->getPlugin();
    $this->expectException(PaymentGatewayException::class);
    $plugin->onReturn($order, $request);
  }

  /**
   * Data provider for testOnReturnException.
   *
   * @return array
   *   An array of test data.
   */
  public static function testOnReturnExceptionDataProvider() {
    $data = [];
    // Request does not contain a remote payment ID.
    $data[] = [
      [
        'type' => 'default',
        'store_id' => 1,
      ],
      new Request(),
    ];
    // Remote payment ID is invalid.
    $data[] = [
      [
        'type' => 'default',
        'store_id' => 1,
      ],
      new Request(['pid' => '0987654321']),
    ];
    return $data;
  }

  /**
   * Tests the payment gateway plugin onNotify() method.
   */
  public function testOnNotify() {
    $this->createOrder($this->getValidOrderValues());
    $this->createPayment($this->getValidPaymentValues());
    $plugin = $this->almaGateway->getPlugin();

    $this->assertEquals(new JsonResponse(), $plugin->onNotify($this->getValidRequest()));
  }

  /**
   * Tests the payment gateway plugin onNotify() method.
   *
   * @dataProvider testOnNotifyExceptionDataProvider
   */
  public function testOnNotifyException(Request $request) {
    $plugin = $this->almaGateway->getPlugin();
    $this->expectException(PaymentGatewayException::class);
    $plugin->onNotify($request);
  }

  /**
   * Data provider for testOnNotifyException.
   *
   * @return array
   *   An array of test data.
   */
  public static function testOnNotifyExceptionDataProvider() {
    $data = [];
    // Request does not contain a remote payment ID.
    $data[] = [
      new Request(),
    ];
    // Remote payment ID is invalid.
    $data[] = [
      new Request(['pid' => '0987654321']),
    ];
    return $data;
  }

  /**
   * Tests the payment gateway plugin validatePayment() method.
   */
  public function testValidatePayment() {
    $this->createOrder($this->getValidOrderValues());
    $payment = $this->createPayment($this->getValidPaymentValues());
    $remote_payment = new RemotePayment([
      'id' => Payments::VALID_PID,
      'purchase_amount' => 2990,
      'state' => RemotePayment::STATE_PAID,
    ]);
    $plugin = $this->almaGateway->getPlugin();
    $this->assertEquals($payment, $plugin->validatePayment($remote_payment));
  }

  /**
   * Tests the payment gateway plugin validatePayment() method.
   *
   * @dataProvider testValidatePaymentExceptionDataProvider
   */
  public function testValidatePaymentException(RemotePayment $remote_payment, array $payment_values, array $order_values) {
    if ($order_values) {
      $this->createOrder($order_values);
    }
    if ($payment_values) {
      $this->createPayment($payment_values);
    }
    $plugin = $this->almaGateway->getPlugin();
    $this->expectException(PaymentGatewayException::class);
    $plugin->validatePayment($remote_payment);
  }

  /**
   * Data provider for testValidatePaymentException.
   *
   * @return array
   *   An array of test data.
   */
  public static function testValidatePaymentExceptionDataProvider() {
    $data = [];
    // Remote payment not matching any payment.
    $data[] = [
      new RemotePayment([
        'id' => '0987654321',
      ]),
      [],
      [],
    ];
    // Matching order is already paid.
    $data[] = [
      new RemotePayment([
        'id' => Payments::VALID_PID,
      ]),
      [
        'type' => 'payment_default',
        'payment_gateway' => 'offsite',
        'remote_id' => Payments::VALID_PID,
        'order_id' => 1,
        'amount' => new Price('29.90', 'EUR'),
        'state' => 'completed',
      ],
      [
        'type' => 'default',
        'store_id' => 1,
        'items' => [
          [
            'type' => 'default',
            'quantity' => 1,
            'unit_price' => new Price('29.90', 'EUR'),
          ],
        ],
      ],
    ];
    // Remote payment state is invalid.
    $data[] = [
      new RemotePayment([
        'id' => Payments::VALID_PID,
        'state' => 'invalid_state',
      ]),
      [
        'type' => 'payment_default',
        'payment_gateway' => 'offsite',
        'remote_id' => Payments::VALID_PID,
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
      ],
      [
        'type' => 'default',
        'store_id' => 1,
        'items' => [
          [
            'type' => 'default',
            'quantity' => 1,
            'unit_price' => new Price('29.90', 'EUR'),
          ],
        ],
      ],
    ];
    // Remote payment amount does not match the payement amount.
    $data[] = [
      new RemotePayment([
        'id' => Payments::VALID_PID,
        'purchase_amount' => 1000,
        'state' => RemotePayment::STATE_PAID,
      ]),
      [
        'type' => 'payment_default',
        'payment_gateway' => 'offsite',
        'remote_id' => Payments::VALID_PID,
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
      ],
      [
        'type' => 'default',
        'store_id' => 1,
        'items' => [
          [
            'type' => 'default',
            'quantity' => 1,
            'unit_price' => new Price('29.90', 'EUR'),
          ],
        ],
      ],
    ];
    // @todo Tests order balance smaller than payment amount.
    return $data;
  }

  /**
   * Tests the payment gateway plugin canRefundPayment() method.
   */
  public function testCanRefundPayment() {
    $this->createOrder($this->getValidOrderValues());
    $payment = $this->createPayment($this->getValidPaymentValues());
    $plugin = $this->almaGateway->getPlugin();
    $this->assertTrue($plugin->canRefundPayment($payment));

    $payment->setRemoteId('0987654321')->save();
    $this->assertFalse($plugin->canRefundPayment($payment));
  }

  /**
   * Tests the payment gateway plugin refundPayment() method.
   */
  public function testRefundPayment() {
    $this->createOrder($this->getValidOrderValues());
    $payment = $this->createPayment($this->getValidPaymentValues());
    $payment->getState()->applyTransitionById('authorize_capture');
    $plugin = $this->almaGateway->getPlugin();

    $plugin->refundPayment($payment, new Price('19.90', 'EUR'));
    $this->assertEquals('partially_refunded', $payment->getState()->getId());
    $this->assertEquals(new Price('19.90', 'EUR'), $payment->getRefundedAmount());

    $plugin->refundPayment($payment, new Price('10.00', 'EUR'));
    $this->assertEquals('refunded', $payment->getState()->getId());
    $this->assertEquals(new Price('29.90', 'EUR'), $payment->getRefundedAmount());
  }

  /**
   * Tests the payment gateway plugin refundPayment() method.
   */
  public function testRefundPaymentException() {
    $this->createOrder($this->getValidOrderValues());
    $payment = $this->createPayment($this->getValidPaymentValues());
    $payment->getState()->applyTransitionById('authorize_capture');
    $plugin = $this->almaGateway->getPlugin();

    $payment->setRemoteId('0987654321')->save();
    $this->expectException(PaymentGatewayException::class);
    $plugin->refundPayment($payment);
  }

}
