<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_alma\Kernel;

use Alma\API\Entities\Payment as RemotePayment;
use Drupal\commerce_alma_test\Api\Endpoints\Payments;
use Drupal\commerce_price\Price;

/**
 * Tests the payment update.
 */
class QueueWorkerTest extends KernelTestBase {

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactoryInterface
   */
  protected $queueFactory;

  /**
   * The queue worker manager.
   *
   * @var \Drupal\Core\Queue\QueueWorkerManagerInterface
   */
  protected $queueWorkerManager;

  /**
   * The Alma payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $almaGateway;

  /**
   * A non Alma payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $altGateway;

  /**
   * The Alma payment gateway with no payments update.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $almaGatewayNoPaymentsUpdate;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->queueFactory = $this->container->get('queue');
    $this->queueWorkerManager = $this->container->get('plugin.manager.queue_worker');

    $this->almaGateway = $this->createGateway([
      'id' => 'offsite',
      'label' => $this->randomString(),
      'plugin' => 'alma',
      'configuration' => [
        'payment_method_types' => ['credit_card'],
        'api_key' => static::TEST_API_KEY,
        'mode' => 'live',
        'fee_plan' => 'general_4_0_0',
      ],
    ]);

    $this->almaGatewayNoPaymentsUpdate = $this->createGateway([
      'id' => 'offsite_no_update',
      'label' => $this->randomString(),
      'plugin' => 'alma',
      'configuration' => [
        'payment_method_types' => ['credit_card'],
        'api_key' => static::TEST_API_KEY,
        'mode' => 'live',
        'fee_plan' => 'general_4_0_0',
        'update_payments' => FALSE,
      ],
    ]);

    $this->altGateway = $this->createGateway([
      'id' => 'alt',
      'label' => $this->randomString(),
      'plugin' => 'manual',
    ]);
  }

  /**
   * Tests update (cron and queue worker).
   *
   * @dataProvider testProcessItemDataProvider
   */
  public function testProcessItem(array $payment_values): void {
    $queue = $this->queueFactory->get('commerce_alma_payment_updater');
    $payment = $this->createPayment($payment_values);
    $queue->createItem($payment);
    $queue_worker = $this->queueWorkerManager->createInstance('commerce_alma_payment_updater');
    $item = $queue->claimItem();
    $queue_worker->processItem($item->data);
    $payment = $this->reloadEntity($payment);
    $this->assertEquals($payment->getRemoteState(), RemotePayment::STATE_PAID);
  }

  /**
   * Data provider for testProcessItem.
   *
   * @return array
   *   The data provider.
   */
  public static function testProcessItemDataProvider(): array {
    $data = [];
    $data[] = [
      [
        'type' => 'payment_default',
        'payment_gateway' => 'offsite',
        'remote_id' => Payments::VALID_PID,
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
        'remote_state' => RemotePayment::STATE_IN_PROGRESS,
      ],
    ];
    return $data;
  }

  /**
   * Tests update (cron and queue worker).
   *
   * @dataProvider testProcessItemAbortDataProvider
   */
  public function testProcessItemAbort(array $payment_values): void {
    $queue = $this->queueFactory->get('commerce_alma_payment_updater');
    $payment = $this->createPayment($payment_values);
    $queue->createItem($payment);
    $queue_worker = $this->queueWorkerManager->createInstance('commerce_alma_payment_updater');
    $item = $queue->claimItem();
    $queue_worker->processItem($item->data);
    $payment = $this->reloadEntity($payment);
    $this->assertEquals($payment->getRemoteState(), RemotePayment::STATE_IN_PROGRESS);
  }

  /**
   * Data provider for testProcessItemAbort.
   *
   * @return array
   *   The data provider.
   */
  public static function testProcessItemAbortDataProvider(): array {
    $data = [];
    // The gateway configuration do not allow payments update.
    $data[] = [
      [
        'type' => 'payment_default',
        'payment_gateway' => 'offsite_no_update',
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
        'remote_state' => RemotePayment::STATE_IN_PROGRESS,
      ],
    ];
    // The gateway is not valid.
    $data[] = [
      [
        'type' => 'payment_default',
        'payment_gateway' => 'alt',
        'amount' => new Price('29.90', 'EUR'),
        'order_id' => 1,
        'remote_state' => RemotePayment::STATE_IN_PROGRESS,
      ],
    ];
    return $data;
  }

}
