<?php

declare(strict_types=1);

namespace Drupal\commerce_asaas;

use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides API call functionalities for the Asaas service.
 */
final class ApiCalls {

  /**
   * Logger interface.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a CreateCustomer object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   Logger factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   */
  public function __construct(LoggerChannelFactoryInterface $logger, RequestStack $requestStack) {
    $this->logger = $logger->get('commerce_asaas');
    $this->requestStack = $requestStack;
  }

  /**
   * Create customer in Asaas API.
   *
   * @param array $configuration
   *   The configuration of the request.
   * @param array $body
   *   The body of the request.
   * @param string $operation
   *   Operation called in the endpoint (customer, payment).
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response of the request.
   */
  private function postRequest($configuration, $body, $operation) {
    $client = new Client();

    $sandbox = $configuration['mode'] == 'test';

    if ($sandbox) {
      $api_key = $configuration['api_key_sandbox'];
      $endpoint = "https://sandbox.asaas.com/api/v3/{$operation}";
    }
    else {
      $api_key = $configuration['api_key'];
      $endpoint = "https://api.asaas.com/{$operation}";
    }

    $response = $client->request('POST', $endpoint, [
      'body' => json_encode($body),
      'headers' => [
        'accept' => 'application/json',
        'access_token' => $api_key,
        'content-type' => 'application/json',
      ],
    ]);

    return $response;
  }

  /**
   * Create customer in Asaas API.
   *
   * @param array $configuration
   *   The configuration of the request.
   * @param \Drupal\user\UserInterface $customer
   *   The customer.
   * @param \Drupal\profile\Entity\ProfileInterface $billing_profile
   *   The billing profile.
   *
   * @return string
   *   The customer ID.
   */
  public function createCustomer($configuration, $customer, $billing_profile) {
    $postal_code = '';
    $additional_name = '';

    $address = $billing_profile->get('address')->first();

    if ($address) {
      $postal_code = $address->postal_code->value ?? '';
      $postal_code = preg_replace('/\D/', '', $postal_code);
      $additional_name = $address->additional_name->value ?? '';
    }

    // Add hyphen in the format XXXXX-XXX.
    if (strlen($postal_code) === 8) {
      $postal_code = preg_replace('/(\d{5})(\d{3})/', '$1-$2', $postal_code);
    }

    $data = [
      'name' => $customer->getDisplayName(),
      'email' => $customer->getEmail(),
      'cpfCnpj' => $billing_profile->cpf->value ?? NULL,
      'mobilePhone' => $billing_profile->field_mobile_phone->value ?? NULL,
      'complement' => $additional_name,
      'postalCode' => $postal_code,
      'externalReference' => $customer->id(),
    ];

    try {
      $response = $this->postRequest($configuration, $data, 'customers');

      if ($response->getStatusCode() == 200) {
        $body = Json::decode($response->getBody());

        return $body['id'];
      }
      else {
        $body = Json::decode($response->getBody());
        $this->logger->error('Error creating customer in Asaas API. Status code: @status, Response: @response', [
          '@status' => $response->getStatusCode(),
          '@response' => $response->getBody()
        ]);
        return NULL;
      }
    }
    catch (RequestException $e) {
      $this->logger->error('Request exception in createCustomer: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Tokenize credit card in Asaas API.
   *
   * @param array $configuration
   *   The configuration of the request.
   * @param array $payment_details
   *   The payment details.
   * @param \Drupal\profile\Entity\ProfileInterface $billing_profile
   *   The billing profile.
   * @param \Drupal\user\UserInterface $customer
   *   The customer.
   * @param string $customer_id
   *   The customer ID from Asaas.
   *
   * @return string
   *   The credit card token.
   */
  public function tokenizeCreditCard($configuration, $payment_details, $billing_profile, $customer, $customer_id = NULL) {
    $postal_code = '';
    $additional_name = '';

    $address = $billing_profile->get('address')->first();

    if ($address) {
      $postal_code = $address->postal_code;
      $postal_code = preg_replace('/\D/', '', $postal_code);
      $additional_name = $address->additional_name;
    }

    // Add hyphen in the format XXXXX-XXX.
    if (strlen($postal_code) === 8) {
      $postal_code = preg_replace('/(\d{5})(\d{3})/', '$1-$2', $postal_code);
    }

    // Use the provided customer_id or try to get it from the user
    if (!$customer_id) {
      $remote_ids = $customer->get('commerce_remote_id');
      if (!$remote_ids->isEmpty()) {
        $customer_id = $remote_ids->getByProvider('commerce_asaas');
      }
    }

    if (!$customer_id) {
      $this->logger->error('No customer ID available for tokenizing credit card');
      return NULL;
    }

    $data = [
      'creditCard' => [
        'holderName' => $payment_details['holder_name'],
        'number' => $payment_details['number'],
        'expiryMonth' => $payment_details['expiration']['month'],
        'expiryYear' => $payment_details['expiration']['year'],
        'ccv' => $payment_details['security_code'],
      ],
      'creditCardHolderInfo' => [
        'name' => $customer->getDisplayName(),
        'email' => $customer->getEmail(),
        'cpfCnpj' => $billing_profile->hasField('cpf') && !$billing_profile->get('cpf')->isEmpty() ? $billing_profile->get('cpf')->value : NULL,
        'postalCode' => $postal_code,
        'addressNumber' => '123',
        'addressComplement' => $additional_name,
        'mobilePhone' => $billing_profile->hasField('mobile_phone') && !$billing_profile->get('mobile_phone')->isEmpty() ? $billing_profile->get('mobile_phone')->value : NULL,
      ],
      'customer' => $customer_id,
      'remoteIp' => $this->requestStack->getCurrentRequest()->getClientIp(),
    ];

    try {
      $response = $this->postRequest($configuration, $data, 'creditCard/tokenizeCreditCard');

      if ($response->getStatusCode() == 200) {
        $body = Json::decode($response->getBody());

        $this->logger->info('Credit card tokenized successfully. Token: @token', ['@token' => $body['creditCardToken']]);
        return $body['creditCardToken'];
      }
      else {
        $body = Json::decode($response->getBody());
        $this->logger->error('Error tokenizing credit card in Asaas API. Status code: @status, Response: @response', [
          '@status' => $response->getStatusCode(),
          '@response' => $response->getBody()
        ]);
        return NULL;
      }
    }
    catch (RequestException $e) {
      $this->logger->error('Request exception in tokenizeCreditCard: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Get customer from Asaas API.
   *
   * @param array $configuration
   *   The configuration of the request.
   * @param string $customer_id
   *   The customer ID.
   *
   * @return \Psr\Http\Message\ResponseInterface
   *   The response of the request.
   */
  public function getCustomer($configuration, $customer_id) {
    $client = new Client();

    $sandbox = $configuration['mode'] == 'test';

    if ($sandbox) {
      $api_key = $configuration['api_key_sandbox'];
      $endpoint = "https://sandbox.asaas.com/api/v3/customers/$customer_id";
    }
    else {
      $api_key = $configuration['api_key'];
      $endpoint = "https://api.asaas.com/customers/$customer_id";
    }

    try {
      $response = $client->request('GET', $endpoint, [
        'headers' => [
          'accept' => 'application/json',
          'access_token' => $api_key,
        ],
      ]);

      if ($response->getStatusCode() == 200) {
        $body = Json::decode($response->getBody());

        return $body['id'];
      }
      else {
        $this->logger->error('Error creating customer in Asaas API. Status code: @status', ['@status' => $response->getStatusCode()]);
        return NULL;
      }
    }
    catch (RequestException $e) {
      $this->logger->error($e->getMessage());
      return NULL;
    }

  }

  /**
   * Create payment in Asaas API.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface $payment
   *   The payment entity.
   * @param array $configuration
   *   The configuration of the request.
   * @param string $customer_id
   *   The customer ID.
   * @param bool $capture
   *   Whether to capture the payment immediately.
   *
   * @return string|null
   *   The payment ID or NULL on failure.
   */
  public function createPayment(PaymentInterface $payment, $configuration, $customer_id, $capture = TRUE) {
    $payment_method = $payment->getPaymentMethod();
    $order = $payment->getOrder();
    $amount = $payment->getAmount();

    // Get payment method type (may be null for offsite payments like PIX)
    $payment_method_type = $payment_method ? $payment_method->getType()->getPluginId() : 'pix';

    $data = [
      'customer' => $customer_id,
      'value' => number_format((float) $amount->getNumber(), 2, '.', ''),
      'dueDate' => date('Y-m-d', strtotime('+3 days')),
      'description' => 'Order #' . $order->id(),
      'externalReference' => $order->id(),
      'remoteIp' => $this->requestStack->getCurrentRequest()->getClientIp(),
    ];

    // Add payment method specific data
    switch ($payment_method_type) {
      case 'credit_card':
        if ($payment_method) {
          $data['billingType'] = 'CREDIT_CARD';
          $data['creditCardToken'] = $payment_method->getRemoteId();
          $data['authorizeOnly'] = !$capture;

          $this->logger->info('Using credit card token: @token', ['@token' => $payment_method->getRemoteId()]);
        }
        break;

      case 'pix':
        $data['billingType'] = 'PIX';
        break;

    }

    try {
      $this->logger->info('Creating payment in Asaas API with data: @data', ['@data' => json_encode($data)]);
      $response = $this->postRequest($configuration, $data, 'payments');

      if ($response->getStatusCode() == 200) {
        $body = Json::decode($response->getBody());
        $this->logger->info('Payment created successfully in Asaas API. Response: @response', ['@response' => json_encode($body)]);
        return $body['id'];
      }
      else {
        $body = Json::decode($response->getBody());
        $this->logger->error('Error creating payment in Asaas API. Status code: @status, Response: @response, Data: @data', [
          '@status' => $response->getStatusCode(),
          '@response' => $response->getBody(),
          '@data' => json_encode($data)
        ]);
        return NULL;
      }
    }
    catch (RequestException $e) {
      $this->logger->error('Request exception in createPayment: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Get PIX QR Code from Asaas API.
   *
   * @param array $configuration
   *   The configuration of the request.
   * @param string $payment_id
   *   The payment ID.
   *
   * @return array|null
   *   The PIX QR Code data or NULL on failure.
   */
  public function getPixQrCode($configuration, $payment_id) {
    $client = new Client();

    $sandbox = $configuration['mode'] == 'test';

    if ($sandbox) {
      $api_key = $configuration['api_key_sandbox'];
      $endpoint = "https://sandbox.asaas.com/api/v3/payments/{$payment_id}/pixQrCode";
    }
    else {
      $api_key = $configuration['api_key'];
      $endpoint = "https://api.asaas.com/payments/{$payment_id}/pixQrCode";
    }

    try {
      $response = $client->request('GET', $endpoint, [
        'headers' => [
          'accept' => 'application/json',
          'access_token' => $api_key,
        ],
      ]);

      if ($response->getStatusCode() == 200) {
        return Json::decode($response->getBody());
      }
      else {
        $this->logger->error('Error getting PIX QR Code from Asaas API. Status code: @status', ['@status' => $response->getStatusCode()]);
        return NULL;
      }
    }
    catch (RequestException $e) {
      $this->logger->error('Request exception in getPixQrCode: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * Get payment from Asaas API.
   *
   * @param array $configuration
   *   The configuration of the request.
   * @param string $payment_id
   *   The payment ID.
   *
   * @return array|null
   *   The payment data or NULL on failure.
   */
  public function getPayment($configuration, $payment_id) {
    $client = new Client();

    $sandbox = $configuration['mode'] == 'test';

    if ($sandbox) {
      $api_key = $configuration['api_key_sandbox'];
      $endpoint = "https://sandbox.asaas.com/api/v3/payments/{$payment_id}";
    }
    else {
      $api_key = $configuration['api_key'];
      $endpoint = "https://api.asaas.com/payments/{$payment_id}";
    }

    try {
      $response = $client->request('GET', $endpoint, [
        'headers' => [
          'accept' => 'application/json',
          'access_token' => $api_key,
        ],
      ]);

      if ($response->getStatusCode() == 200) {
        $body = Json::decode($response->getBody());
        $this->logger->info('Payment data retrieved from Asaas API: @data', ['@data' => json_encode($body)]);
        return $body;
      }
      else {
        $this->logger->error('Error getting payment from Asaas API. Status code: @status, Response: @response', [
          '@status' => $response->getStatusCode(),
          '@response' => $response->getBody()
        ]);
        return NULL;
      }
    }
    catch (RequestException $e) {
      $this->logger->error($e->getMessage());
      return NULL;
    }
  }

  /**
   * Cancel payment in Asaas API.
   *
   * @param array $configuration
   *   The configuration of the request.
   * @param string $payment_id
   *   The payment ID.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function cancelPayment($configuration, $payment_id) {
    $client = new Client();

    $sandbox = $configuration['mode'] == 'test';

    if ($sandbox) {
      $api_key = $configuration['api_key_sandbox'];
      $endpoint = "https://sandbox.asaas.com/api/v3/payments/{$payment_id}/cancel";
    }
    else {
      $api_key = $configuration['api_key'];
      $endpoint = "https://api.asaas.com/payments/{$payment_id}/cancel";
    }

    try {
      $response = $client->request('POST', $endpoint, [
        'headers' => [
          'accept' => 'application/json',
          'access_token' => $api_key,
        ],
      ]);

      if ($response->getStatusCode() == 200) {
        return TRUE;
      }
      else {
        $this->logger->error('Error canceling payment in Asaas API. Status code: @status', ['@status' => $response->getStatusCode()]);
        return FALSE;
      }
    }
    catch (RequestException $e) {
      $this->logger->error($e->getMessage());
      return FALSE;
    }
  }

  /**
   * Refund payment in Asaas API.
   *
   * @param array $configuration
   *   The configuration of the request.
   * @param string $payment_id
   *   The payment ID.
   * @param float $value
   *   The refund value.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function refundPayment($configuration, $payment_id, $value = NULL) {
    $client = new Client();

    $sandbox = $configuration['mode'] == 'test';

    if ($sandbox) {
      $api_key = $configuration['api_key_sandbox'];
      $endpoint = "https://sandbox.asaas.com/api/v3/payments/{$payment_id}/refund";
    }
    else {
      $api_key = $configuration['api_key'];
      $endpoint = "https://api.asaas.com/payments/{$payment_id}/refund";
    }

    $data = [];
    if ($value !== NULL) {
      $data['value'] = number_format((float) $value, 2, '.', '');
    }

    try {
      $response = $client->request('POST', $endpoint, [
        'body' => json_encode($data),
        'headers' => [
          'accept' => 'application/json',
          'access_token' => $api_key,
          'content-type' => 'application/json',
        ],
      ]);

      if ($response->getStatusCode() == 200) {
        return TRUE;
      }
      else {
        $this->logger->error('Error refunding payment in Asaas API. Status code: @status', ['@status' => $response->getStatusCode()]);
        return FALSE;
      }
    }
    catch (RequestException $e) {
      $this->logger->error($e->getMessage());
      return FALSE;
    }
  }

}
