<?php

namespace Drupal\commerce_asaas\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_asaas\ApiCalls;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for checking payment status.
 */
class PaymentStatusController extends ControllerBase {

  /**
   * The API calls service.
   *
   * @var \Drupal\commerce_asaas\ApiCalls
   */
  protected $apiCalls;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new PaymentStatusController object.
   *
   * @param \Drupal\commerce_asaas\ApiCalls $api_calls
   *   The API calls service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(ApiCalls $api_calls, LoggerChannelFactoryInterface $logger_factory) {
    $this->apiCalls = $api_calls;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commerce_asaas.api_calls'),
      $container->get('logger.factory')
    );
  }

  /**
   * Checks payment status for PIX payments.
   *
   * @param string $payment_id
   *   The payment ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function checkPaymentStatus($payment_id, Request $request) {
    $logger = $this->loggerFactory->get('commerce_asaas');

    try {
      // Get payment gateway configuration
      $payment_gateway = $this->entityTypeManager()
        ->getStorage('commerce_payment_gateway')
        ->loadByProperties(['plugin' => 'asaas_pix']);

      if (empty($payment_gateway)) {
        return new JsonResponse(['error' => 'Payment gateway not found'], 404);
      }

      $gateway = reset($payment_gateway);
      $configuration = $gateway->getPlugin()->getConfiguration();

      // Get payment data from Asaas
      $payment_data = $this->apiCalls->getPayment($configuration, $payment_id);

      if (!$payment_data) {
        return new JsonResponse(['error' => 'Payment not found'], 404);
      }

      $status = $payment_data['status'] ?? 'unknown';
      $is_paid = $status === 'RECEIVED' || $status === 'CONFIRMED';

      // Check if PIX QR Code is available
      $pix_data = $this->apiCalls->getPixQrCode($configuration, $payment_id);
      $pix_available = $pix_data && isset($pix_data['encodedImage']) && isset($pix_data['payload']);

      $logger->info('Payment status check - ID: @id, Status: @status, PIX Available: @pix', [
        '@id' => $payment_id,
        '@status' => $status,
        '@pix' => $pix_available ? 'YES' : 'NO'
      ]);

      return new JsonResponse([
        'payment_id' => $payment_id,
        'status' => $status,
        'is_paid' => $is_paid,
        'pix_available' => $pix_available,
        'amount' => $payment_data['value'] ?? 0,
        'due_date' => $payment_data['dueDate'] ?? NULL,
      ]);
    }
    catch (\Exception $e) {
      $logger->error('Error checking payment status: @error', ['@error' => $e->getMessage()]);
      return new JsonResponse(['error' => 'Internal server error'], 500);
    }
  }

}
