<?php

namespace Drupal\commerce_asaas\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\PaymentStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Controller for handling Asaas webhooks.
 */
class WebhookController extends ControllerBase {

  /**
   * The payment storage.
   *
   * @var \Drupal\commerce_payment\PaymentStorageInterface
   */
  protected $paymentStorage;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new WebhookController object.
   *
   * @param \Drupal\commerce_payment\PaymentStorageInterface $payment_storage
   *   The payment storage.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(PaymentStorageInterface $payment_storage, LoggerChannelFactoryInterface $logger_factory) {
    $this->paymentStorage = $payment_storage;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')->getStorage('commerce_payment'),
      $container->get('logger.factory')
    );
  }

  /**
   * Handles webhook notifications from Asaas.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function handleWebhook(Request $request) {
    $logger = $this->loggerFactory->get('commerce_asaas');

    try {
      $payload = $request->getContent();
      $data = json_decode($payload, TRUE);

      if (!$data) {
        $logger->error('Invalid JSON payload received from Asaas webhook.');
        return new Response('Invalid JSON', 400);
      }

      $logger->info('Received webhook from Asaas: @data', ['@data' => $payload]);

      // Verify webhook signature if configured
      if (!$this->verifyWebhookSignature($request, $payload)) {
        $logger->error('Invalid webhook signature from Asaas.');
        return new Response('Invalid signature', 403);
      }

      // Process the webhook based on event type
      $event_type = $data['event'] ?? '';
      $payment_data = $data['payment'] ?? [];

      $logger->info('Processing webhook event: @event for payment: @payment_id', [
        '@event' => $event_type,
        '@payment_id' => $payment_data['id'] ?? 'unknown',
      ]);

      switch ($event_type) {
        case 'PAYMENT_CREATED':
          $this->handlePaymentCreated($payment_data);
          break;

        case 'PAYMENT_AWAITING_PAYMENT':
          $this->handlePaymentAwaitingPayment($payment_data);
          break;

        case 'PAYMENT_RECEIVED':
          $this->handlePaymentReceived($payment_data);
          break;

        case 'PAYMENT_CONFIRMED':
          $this->handlePaymentReceived($payment_data);
          break;

        case 'PAYMENT_OVERDUE':
          $this->handlePaymentOverdue($payment_data);
          break;

        case 'PAYMENT_DELETED':
          $this->handlePaymentDeleted($payment_data);
          break;

        case 'PAYMENT_RESTORED':
          $this->handlePaymentRestored($payment_data);
          break;

        case 'PAYMENT_REFUNDED':
          $this->handlePaymentRefunded($payment_data);
          break;

        case 'PAYMENT_RECEIVED_IN_CASH_UNDONE':
          $this->handlePaymentReceivedInCashUndone($payment_data);
          break;

        case 'PAYMENT_CHARGEBACK_REQUESTED':
          $this->handlePaymentChargebackRequested($payment_data);
          break;

        case 'PAYMENT_CHARGEBACK_DISPUTE':
          $this->handlePaymentChargebackDispute($payment_data);
          break;

        case 'PAYMENT_AWAITING_RISK_ANALYSIS':
          $this->handlePaymentAwaitingRiskAnalysis($payment_data);
          break;

        default:
          $logger->warning('Unknown webhook event type: @event', ['@event' => $event_type]);
      }

      return new JsonResponse(['status' => 'success']);
    }
    catch (\Exception $e) {
      $logger->error('Error processing Asaas webhook: @error', ['@error' => $e->getMessage()]);
      return new Response('Internal server error', 500);
    }
  }

  /**
   * Verifies webhook signature.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param string $payload
   *   The request payload.
   *
   * @return bool
   *   TRUE if signature is valid, FALSE otherwise.
   */
  protected function verifyWebhookSignature(Request $request, $payload) {
    // For now, we'll skip signature verification
    // In production, you should implement proper signature verification
    // using the webhook secret from Asaas configuration
    return TRUE;
  }

  /**
   * Handles payment created event.
   */
  protected function handlePaymentCreated($payment_data) {
    $logger = $this->loggerFactory->get('commerce_asaas');
    $logger->info('Payment created: @id', ['@id' => $payment_data['id']]);
  }

  /**
   * Handles payment awaiting payment event.
   */
  protected function handlePaymentAwaitingPayment($payment_data) {
    $this->updatePaymentStatus($payment_data, 'pending');
  }

  /**
   * Handles payment received event.
   */
  protected function handlePaymentReceived($payment_data) {
    $this->updatePaymentStatus($payment_data, 'completed');
    $this->updateOrderStatus($payment_data, 'completed');
  }

  /**
   * Handles payment overdue event.
   */
  protected function handlePaymentOverdue($payment_data) {
    $this->updatePaymentStatus($payment_data, 'overdue');
  }

  /**
   * Handles payment deleted event.
   */
  protected function handlePaymentDeleted($payment_data) {
    $this->updatePaymentStatus($payment_data, 'cancelled');
  }

  /**
   * Handles payment restored event.
   */
  protected function handlePaymentRestored($payment_data) {
    $this->updatePaymentStatus($payment_data, 'pending');
  }

  /**
   * Handles payment refunded event.
   */
  protected function handlePaymentRefunded($payment_data) {
    $this->updatePaymentStatus($payment_data, 'refunded');
  }

  /**
   * Handles payment received in cash undone event.
   */
  protected function handlePaymentReceivedInCashUndone($payment_data) {
    $this->updatePaymentStatus($payment_data, 'pending');
  }

  /**
   * Handles payment chargeback requested event.
   */
  protected function handlePaymentChargebackRequested($payment_data) {
    $this->updatePaymentStatus($payment_data, 'chargeback');
  }

  /**
   * Handles payment chargeback dispute event.
   */
  protected function handlePaymentChargebackDispute($payment_data) {
    $this->updatePaymentStatus($payment_data, 'chargeback');
  }

  /**
   * Handles payment awaiting risk analysis event.
   */
  protected function handlePaymentAwaitingRiskAnalysis($payment_data) {
    $this->updatePaymentStatus($payment_data, 'pending');
  }

  /**
   * Updates payment status based on webhook data.
   *
   * @param array $payment_data
   *   The payment data from webhook.
   * @param string $status
   *   The new status.
   */
  protected function updatePaymentStatus($payment_data, $status) {
    $logger = $this->loggerFactory->get('commerce_asaas');

    $payment_id = $payment_data['id'];
    $payments = $this->paymentStorage->loadByProperties(['remote_id' => $payment_id]);

    if (empty($payments)) {
      $logger->warning('Payment not found for remote ID: @id', ['@id' => $payment_id]);
      return;
    }

    $payment = reset($payments);

    // Map Asaas status to Commerce status
    $status_mapping = [
      'pending' => 'pending',
      'completed' => 'completed',
      'overdue' => 'overdue',
      'cancelled' => 'cancelled',
      'refunded' => 'refunded',
      'chargeback' => 'chargeback',
    ];

    if (isset($status_mapping[$status])) {
      $payment->setState($status_mapping[$status]);
      $payment->save();

      $logger->info('Updated payment @id status to @status', [
        '@id' => $payment_id,
        '@status' => $status,
      ]);
    }
  }

  /**
   * Updates order status based on payment status.
   *
   * @param array $payment_data
   *   The payment data from webhook.
   * @param string $status
   *   The new status.
   */
  protected function updateOrderStatus($payment_data, $status) {
    $logger = $this->loggerFactory->get('commerce_asaas');

    $payment_id = $payment_data['id'];
    $payments = $this->paymentStorage->loadByProperties(['remote_id' => $payment_id]);

    if (empty($payments)) {
      $logger->warning('Payment not found for remote ID: @id when updating order', ['@id' => $payment_id]);
      return;
    }

    $payment = reset($payments);
    $order = $payment->getOrder();

    if (!$order) {
      $logger->warning('Order not found for payment @id', ['@id' => $payment_id]);
      return;
    }

    // Map payment status to order transitions
    $order_transition_mapping = [
      'completed' => 'complete',
      'pending' => 'place',
      'cancelled' => 'cancel',
      'refunded' => 'cancel',
    ];

    if (isset($order_transition_mapping[$status])) {
      $transition_id = $order_transition_mapping[$status];

      // Check if the transition is available
      if ($order->getState()->hasTransition($transition_id)) {
        $order->getState()->applyTransitionById($transition_id);
        $order->save();

        $logger->info('Updated order @order_id status to @status based on payment @payment_id using transition @transition', [
          '@order_id' => $order->id(),
          '@status' => $status,
          '@payment_id' => $payment_id,
          '@transition' => $transition_id,
        ]);
      } else {
        $logger->warning('Transition @transition not available for order @order_id in state @current_state', [
          '@transition' => $transition_id,
          '@order_id' => $order->id(),
          '@current_state' => $order->getState()->getId(),
        ]);
      }
    }
  }

}
