<?php

namespace Drupal\commerce_asaas\Plugin\Commerce\PaymentGateway;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_asaas\PluginForm\OffsiteRedirect\PaymentOffsiteForm;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Off-site Redirect payment gateway.
 */
#[CommercePaymentGateway(
  id: "asaas_offsite_redirect",
  label: new TranslatableMarkup("Asaas Off-site redirect"),
  display_label: new TranslatableMarkup("All Payment Methods"),
  forms: [
    "offsite-payment" => PaymentOffsiteForm::class,
  ],
  payment_method_types: ["credit_card"],
  credit_card_types: [
    "amex",
    "dinersclub",
    "discover",
    "jcb",
    "maestro",
    "mastercard",
    "visa",
  ],
  requires_billing_information: FALSE,
)]
class OffsiteRedirect extends OffsitePaymentGatewayBase {

  /**
   * {@inheritdoc}
   */
  public function getPaymentMethodTypes() {
    // Ensure paymentMethodTypes is initialized
    if (is_null($this->paymentMethodTypes)) {
      $this->initializePaymentMethodTypes();
    }

    // Return the actual plugin instances
    return $this->paymentMethodTypes;
  }

  /**
   * {@inheritdoc}
   */
  public function getPaymentType() {
    // Ensure paymentType is initialized
    if (is_null($this->paymentType)) {
      $this->initializePaymentType();
    }

    return $this->paymentType;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'redirect_method' => 'post',
      'payment_method_types' => ['credit_card'],
    ] + parent::defaultConfiguration();
  }

  /**
   * Initialize payment method types if not already done.
   */
  protected function initializePaymentMethodTypes() {
    if (is_null($this->paymentMethodTypes)) {
      $this->paymentMethodTypes = [];
      $payment_method_type_manager = \Drupal::service('plugin.manager.commerce_payment_method_type');
      $plugin_definition = $this->getPluginDefinition();

      foreach ($plugin_definition['payment_method_types'] as $plugin_id) {
        $this->paymentMethodTypes[$plugin_id] = $payment_method_type_manager->createInstance($plugin_id);
      }
    }
  }

  /**
   * Initialize payment type if not already done.
   */
  protected function initializePaymentType() {
    if (is_null($this->paymentType)) {
      $payment_type_manager = \Drupal::service('plugin.manager.commerce_payment_type');
      $plugin_definition = $this->getPluginDefinition();
      $this->paymentType = $payment_type_manager->createInstance($plugin_definition['payment_type']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Ensure paymentMethodTypes is initialized before calling parent
    if (is_null($this->paymentMethodTypes)) {
      $this->initializePaymentMethodTypes();
    }

    $form = parent::buildConfigurationForm($form, $form_state);

    // A real gateway would always know which redirect method should be used,
    // it's made configurable here for test purposes.
    $form['redirect_method'] = [
      '#type' => 'radios',
      '#title' => $this->t('Redirect method'),
      '#options' => [
        'get' => $this->t('Redirect via GET (302 header)'),
        'post' => $this->t('Redirect via POST (automatic)'),
        'post_manual' => $this->t('Redirect via POST (manual)'),
      ],
      '#default_value' => $this->configuration['redirect_method'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['redirect_method'] = $values['redirect_method'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    // @todo Add examples of request validation.
    // Note: Since requires_billing_information is FALSE, the order is
    // not guaranteed to have a billing profile. Confirm that
    // $order->getBillingProfile() is not NULL before trying to use it.
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payment = $payment_storage->create([
      'state' => 'completed',
      'amount' => $order->getBalance(),
      'payment_gateway' => $this->parentEntity->id(),
      'order_id' => $order->id(),
      'remote_id' => $request->query->get('txn_id'),
      'remote_state' => $request->query->get('payment_status'),
    ]);
    $payment->save();
  }

}
