<?php

namespace Drupal\commerce_asaas\Plugin\Commerce\PaymentGateway;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\CreditCard;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\HardDeclineException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OnsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsZeroBalanceOrderInterface;
use Drupal\commerce_payment\PluginForm\PaymentMethodEditForm;
use Drupal\commerce_asaas\PluginForm\Onsite\PaymentMethodAddForm;
use Drupal\commerce_asaas\ApiCalls;
use Drupal\commerce_price\Price;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the On-site payment gateway.
 */
#[CommercePaymentGateway(
  id: "asaas_onsite_credit_card",
  label: new TranslatableMarkup("Asaas Credit Card On-site"),
  display_label: new TranslatableMarkup("Credit Card"),
  forms: [
    "add-payment-method" => PaymentMethodAddForm::class,
    "edit-payment-method" => PaymentMethodEditForm::class,
  ],
  payment_method_types: ["credit_card"],
  credit_card_types: [
    "amex", "dinersclub", "discover", "jcb", "maestro", "mastercard", "visa",
  ],
  requires_billing_information: FALSE,
)]
class OnsiteCreditCard extends OnsitePaymentGatewayBase implements OnsiteCreditCardInterface, SupportsZeroBalanceOrderInterface {

  /**
   * The API calls service.
   *
   * @var \Drupal\commerce_asaas\ApiCalls
   */
  protected $apiCalls;

  /**
   * {@inheritdoc}
   */
  public function getPaymentMethodTypes() {
    // Ensure paymentMethodTypes is initialized
    if (is_null($this->paymentMethodTypes)) {
      $this->initializePaymentMethodTypes();
    }

    // Return the actual plugin instances
    return $this->paymentMethodTypes;
  }

  /**
   * {@inheritdoc}
   */
  public function getPaymentType() {
    // Ensure paymentType is initialized
    if (is_null($this->paymentType)) {
      $this->initializePaymentType();
    }

    return $this->paymentType;
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ApiCalls $api_calls) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->apiCalls = $api_calls;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('commerce_asaas.api_calls')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'api_key' => '',
      'api_key_sandbox' => '',
      'endpoint' => 'https://api.asaas.com/',
      'endpoint_sandbox' => 'https://sandbox.asaas.com/api/v3/',
      'payment_method_types' => ['credit_card'],
    ] + parent::defaultConfiguration();
  }

  /**
   * Initialize payment method types if not already done.
   */
  protected function initializePaymentMethodTypes() {
    if (is_null($this->paymentMethodTypes)) {
      $this->paymentMethodTypes = [];
      $payment_method_type_manager = \Drupal::service('plugin.manager.commerce_payment_method_type');
      $plugin_definition = $this->getPluginDefinition();

      foreach ($plugin_definition['payment_method_types'] as $plugin_id) {
        $this->paymentMethodTypes[$plugin_id] = $payment_method_type_manager->createInstance($plugin_id);
      }
    }
  }

  /**
   * Initialize payment type if not already done.
   */
  protected function initializePaymentType() {
    if (is_null($this->paymentType)) {
      $payment_type_manager = \Drupal::service('plugin.manager.commerce_payment_type');
      $plugin_definition = $this->getPluginDefinition();
      $this->paymentType = $payment_type_manager->createInstance($plugin_definition['payment_type']);
    }
  }

  /**
   * Get remote customer ID safely, handling cases where parentEntity is not available.
   */
  protected function getRemoteCustomerIdSafe($account) {
    $remote_id = NULL;
    if (!$account->isAnonymous()) {
      // Use plugin_id if parentEntity is not available
      $gateway_id = $this->parentEntity ? $this->parentEntity->id() : $this->pluginId;
      $provider = $gateway_id . '|' . $this->getMode();
      /** @var \Drupal\commerce\Plugin\Field\FieldType\RemoteIdFieldItemListInterface $remote_ids */
      $remote_ids = $account->get('commerce_remote_id');
      $remote_id = $remote_ids->getByProvider($provider);
      // Gateways used to key customer IDs by module name, migrate that data.
      if (!$remote_id) {
        $remote_id = $remote_ids->getByProvider('commerce_asaas');
      }
    }
    return $remote_id;
  }

  /**
   * Set remote customer ID safely, handling cases where parentEntity is not available.
   */
  protected function setRemoteCustomerIdSafe($account, $remote_id) {
    if (!$account->isAnonymous()) {
      // Use plugin_id if parentEntity is not available
      $gateway_id = $this->parentEntity ? $this->parentEntity->id() : $this->pluginId;
      /** @var \Drupal\commerce\Plugin\Field\FieldType\RemoteIdFieldItemListInterface $remote_ids */
      $remote_ids = $account->get('commerce_remote_id');
      $remote_ids->setByProvider($gateway_id . '|' . $this->getMode(), $remote_id);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Ensure paymentMethodTypes is initialized before calling parent
    if (is_null($this->paymentMethodTypes)) {
      $this->initializePaymentMethodTypes();
    }

    $form = parent::buildConfigurationForm($form, $form_state);

    // Example credential. Also needs matching schema in
    // config/schema/$your_module.schema.yml.
    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live API key'),
      '#size' => 255,
      '#maxlength' => 255,
      '#default_value' => $this->configuration['api_key'],
      '#states' => [
        'required' => [
          ':input[name="configuration[asaas_onsite_credit_card][mode]"]' => [
            'value' => 'live',
          ],
        ],
      ],
    ];

    $form['api_key_sandbox'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sandbox API key'),
      '#size' => 255,
      '#maxlength' => 255,
      '#default_value' => $this->configuration['api_key_sandbox'],
      '#states' => [
        'required' => [
          ':input[name="configuration[asaas_onsite_credit_card][mode]"]' => [
            'value' => 'test',
          ],
        ],
      ],
    ];

    $form['endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live endpoint'),
      '#default_value' => $this->configuration['endpoint'],
    ];

    $form['endpoint_sandbox'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sandbox endpoint'),
      '#default_value' => $this->configuration['endpoint_sandbox'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $configuration = $form_state->getValue('configuration');
    $endpoint = $configuration['asaas_onsite_credit_card']['endpoint'];
    $endpoint_sandbox = $configuration['asaas_onsite_credit_card']['endpoint_sandbox'];

    // Validate if it is a valid URL.
    if (!filter_var($endpoint, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('endpoint', $this->t('The live endpoint entered is not valid. Please provide a valid URL.'));
    }

    // Validate if it is a valid URL.
    if (!filter_var($endpoint_sandbox, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('endpoint_sandbox', $this->t('The sandbox endpoint entered is not valid. Please provide a valid URL.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['api_key'] = $values['api_key'];
      $this->configuration['api_key_sandbox'] = $values['api_key_sandbox'];
      $this->configuration['endpoint'] = $values['endpoint'];
      $this->configuration['endpoint_sandbox'] = $values['endpoint_sandbox'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->assertPaymentState($payment, ['new']);
    $payment_method = $payment->getPaymentMethod();
    $this->assertPaymentMethod($payment_method);

    $owner = $payment_method->getOwner();
    $billing_profile = $payment_method->getBillingProfile();

    // Get or create customer.
    $customer_id = '';
    if ($owner && !$owner->isAnonymous()) {
      $customer_id = $this->getRemoteCustomerIdSafe($owner);

      if (!$customer_id) {
        \Drupal::logger('commerce_asaas')->info('Creating new customer for user @uid', ['@uid' => $owner->id()]);
        $customer_id = $this->apiCalls->createCustomer($this->configuration, $owner, $billing_profile);
        if ($customer_id) {
          $this->setRemoteCustomerIdSafe($owner, $customer_id);
          $owner->save();
          \Drupal::logger('commerce_asaas')->info('Customer created with ID: @customer_id', ['@customer_id' => $customer_id]);
        } else {
          \Drupal::logger('commerce_asaas')->error('Failed to create customer for user @uid', ['@uid' => $owner->id()]);
        }
      } else {
        \Drupal::logger('commerce_asaas')->info('Using existing customer ID: @customer_id', ['@customer_id' => $customer_id]);
      }
    }

    if (!$customer_id) {
      throw new \InvalidArgumentException('Customer ID is required for payment processing.');
    }

    // Check if we need to regenerate the token (if it's expired or invalid)
    $payment_method_remote_id = $payment_method->getRemoteId();

    // For non-reusable payment methods or if token is missing, regenerate it
    if (!$payment_method->isReusable() || !$payment_method_remote_id) {
      \Drupal::logger('commerce_asaas')->info('Regenerating credit card token for payment method @id', ['@id' => $payment_method->id()]);

      // Get payment details from the form (we need to store them temporarily)
      // For now, we'll try to use the existing token, but this needs to be improved
    }

    // Log the payment method details before creating payment
    \Drupal::logger('commerce_asaas')->info('Creating payment with customer ID: @customer_id, payment method remote ID: @remote_id', [
      '@customer_id' => $customer_id,
      '@remote_id' => $payment_method_remote_id
    ]);

    $remote_id = $this->apiCalls->createPayment($payment, $this->configuration, $customer_id, $capture);

    if (!$remote_id) {
      throw new \Exception('Failed to create payment in Asaas API.');
    }

    $next_state = $capture ? 'completed' : 'authorization';
    $payment->setState($next_state);
    $payment->setRemoteId($remote_id);
    $payment->setAvsResponseCode('A');

    if (!$payment_method->card_type->isEmpty()) {
      $avs_response_code_label = $this->buildAvsResponseCodeLabel('A', $payment_method->card_type->value);
      $payment->setAvsResponseCodeLabel($avs_response_code_label);
    }

    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, ?Price $amount = NULL) {
    $this->assertPaymentState($payment, ['authorization']);
    $amount = $amount ?: $payment->getAmount();

    // For Asaas, we need to create a new payment with the same amount
    // since they don't have a separate capture endpoint.
    $success = $this->apiCalls->createPayment($payment, $this->configuration, $this->getRemoteCustomerIdSafe($payment->getPaymentMethod()->getOwner()), TRUE);

    if (!$success) {
      throw new \Exception('Failed to capture payment in Asaas API.');
    }

    $payment->setState('completed');
    $payment->setAmount($amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment) {
    $this->assertPaymentState($payment, ['authorization']);

    $remote_id = $payment->getRemoteId();

    $success = $this->apiCalls->cancelPayment($this->configuration, $remote_id);

    if (!$success) {
      throw new \Exception('Failed to void payment in Asaas API.');
    }

    $payment->setState('authorization_voided');
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, ?Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    $remote_id = $payment->getRemoteId();

    $success = $this->apiCalls->refundPayment($this->configuration, $remote_id, $amount->getNumber());

    if (!$success) {
      throw new \Exception('Failed to refund payment in Asaas API.');
    }

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);

    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->setState('partially_refunded');
    }
    else {
      $payment->setState('refunded');
    }

    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function createPaymentMethod(PaymentMethodInterface $payment_method, array $payment_details) {

    $required_keys = [
      // The expected keys are payment gateway specific and usually match
      // the PaymentMethodAddForm form elements. They are expected to be valid.
      'type', 'number', 'expiration',
    ];
    foreach ($required_keys as $required_key) {
      if (empty($payment_details[$required_key])) {
        throw new \InvalidArgumentException(sprintf('$payment_details must contain the %s key.', $required_key));
      }
    }

    // Add details to payment method before declining so that these details are
    // available to the FailedPaymentEvent.
    $payment_method->card_type = $payment_details['type'];
    // Only the last 4 numbers are safe to store.
    $payment_method->card_number = substr($payment_details['number'], -4);
    $payment_method->card_exp_month = $payment_details['expiration']['month'];
    $payment_method->card_exp_year = $payment_details['expiration']['year'];

    // Add a built in test for testing decline exceptions.
    // Note: Since requires_billing_information is FALSE, the payment method
    // is not guaranteed to have a billing profile. Confirm that
    // $payment_method->getBillingProfile() is not NULL before trying to use it.
    if ($billing_profile = $payment_method->getBillingProfile()) {
      /** @var \Drupal\address\Plugin\Field\FieldType\AddressItem $billing_address */
      $billing_address = $billing_profile->get('address')->first();
      if (strlen($billing_address->getPostalCode()) < 8) {
        throw HardDeclineException::createForPayment($payment_method, $this->t('The postal code is wrong. It must be in the format 99999-999.'));
      }
    }

    // If the remote API needs a remote customer to be created.
    $owner = $payment_method->getOwner();
    if ($owner && !$owner->isAnonymous()) {
      // Use plugin_id instead of parentEntity->id() if parentEntity is not available
      $customer_id = $this->getRemoteCustomerIdSafe($owner);

      if (!$customer_id) {
        $customer_id = $this->apiCalls->createCustomer($this->configuration, $owner, $billing_profile);
        $this->setRemoteCustomerIdSafe($owner, $customer_id);
        $owner->save();
      }
      else {
        $customer = $this->apiCalls->getCustomer($this->configuration, $customer_id);

        if (!$customer) {
          $customer_id = $this->apiCalls->createCustomer($this->configuration, $owner, $billing_profile);
          $this->setRemoteCustomerIdSafe($owner, $customer_id);
          $owner->save();
        }
      }
    }

    // Perform the create request here, throw an exception if it fails.
    // See \Drupal\commerce_payment\Exception for the available exceptions.
    // You might need to do different API requests based on whether the
    // payment method is reusable: $payment_method->isReusable().
    // Non-reusable payment methods usually have an expiration timestamp.
    $expires = CreditCard::calculateExpirationTimestamp($payment_details['expiration']['month'], $payment_details['expiration']['year']);
    // Get customer ID for tokenization
    $customer_id = '';
    if ($owner && !$owner->isAnonymous()) {
      $customer_id = $this->getRemoteCustomerIdSafe($owner);

      if (!$customer_id) {
        $customer_id = $this->apiCalls->createCustomer($this->configuration, $owner, $billing_profile);
        if ($customer_id) {
          $this->setRemoteCustomerIdSafe($owner, $customer_id);
          $owner->save();
        }
      }
    }

    // The remote ID returned by the request.
    $remote_id = $this->apiCalls->tokenizeCreditCard($this->configuration, $payment_details, $billing_profile, $owner, $customer_id);

    if ($remote_id) {
      \Drupal::logger('commerce_asaas')->info('Credit card token generated: @token', ['@token' => $remote_id]);
      $payment_method->setRemoteId($remote_id);
      $payment_method->setExpiresTime($expires);
      $payment_method->save();
    } else {
      \Drupal::logger('commerce_asaas')->error('Failed to generate credit card token');
      throw new \Exception('Failed to generate credit card token');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    // Delete the remote record here, throw an exception if it fails.
    // See \Drupal\commerce_payment\Exception for the available exceptions.
    // Delete the local entity.
    $payment_method->delete();
  }

  /**
   * {@inheritdoc}
   */
  public function updatePaymentMethod(PaymentMethodInterface $payment_method) {
    // Note: Since requires_billing_information is FALSE, the payment method
    // is not guaranteed to have a billing profile. Confirm thatt
    // $payment_method->getBillingProfile() is not NULL before trying to use it.
    //
    // Perform the update request here, throw an exception if it fails.
    // See \Drupal\commerce_payment\Exception for the available exceptions.
  }

  /**
   * {@inheritdoc}
   */
  public function buildAvsResponseCodeLabel($avs_response_code, $card_type) {
    if ($card_type == 'dinersclub' || $card_type == 'jcb') {
      if ($avs_response_code == 'A') {
        return $this->t('Approved.');
      }
      return NULL;
    }
    return parent::buildAvsResponseCodeLabel($avs_response_code, $card_type);
  }

}
