<?php

namespace Drupal\commerce_asaas\Plugin\Commerce\PaymentGateway;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_asaas\PluginForm\OffsiteRedirect\PixPaymentForm;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the PIX payment gateway.
 */
#[CommercePaymentGateway(
  id: "asaas_pix",
  label: new TranslatableMarkup("Asaas PIX"),
  display_label: new TranslatableMarkup("PIX"),
  forms: [
    "offsite-payment" => PixPaymentForm::class,
  ],
  payment_method_types: ["pix"],
  requires_billing_information: TRUE,
)]
class Pix extends OffsitePaymentGatewayBase {

  /**
   * {@inheritdoc}
   */
  public function getPaymentMethodTypes() {
    // Ensure paymentMethodTypes is initialized
    if (is_null($this->paymentMethodTypes)) {
      $this->initializePaymentMethodTypes();
    }

    // Return the actual plugin instances
    return $this->paymentMethodTypes;
  }

  /**
   * {@inheritdoc}
   */
  public function getPaymentType() {
    // Ensure paymentType is initialized
    if (is_null($this->paymentType)) {
      $this->initializePaymentType();
    }

    return $this->paymentType;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'api_key' => '',
      'api_key_sandbox' => '',
      'endpoint' => 'https://api.asaas.com/',
      'endpoint_sandbox' => 'https://sandbox.asaas.com/api/v3/',
      'payment_method_types' => ['pix'],
    ] + parent::defaultConfiguration();
  }

  /**
   * Initialize payment method types if not already done.
   */
  protected function initializePaymentMethodTypes() {
    if (is_null($this->paymentMethodTypes)) {
      $this->paymentMethodTypes = [];
      $payment_method_type_manager = \Drupal::service('plugin.manager.commerce_payment_method_type');
      $plugin_definition = $this->getPluginDefinition();

      foreach ($plugin_definition['payment_method_types'] as $plugin_id) {
        $this->paymentMethodTypes[$plugin_id] = $payment_method_type_manager->createInstance($plugin_id);
      }
    }
  }

  /**
   * Initialize payment type if not already done.
   */
  protected function initializePaymentType() {
    if (is_null($this->paymentType)) {
      $payment_type_manager = \Drupal::service('plugin.manager.commerce_payment_type');
      $plugin_definition = $this->getPluginDefinition();
      $this->paymentType = $payment_type_manager->createInstance($plugin_definition['payment_type']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // Ensure paymentMethodTypes is initialized before calling parent
    if (is_null($this->paymentMethodTypes)) {
      $this->initializePaymentMethodTypes();
    }

    $form = parent::buildConfigurationForm($form, $form_state);

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live API key'),
      '#size' => 255,
      '#maxlength' => 255,
      '#default_value' => $this->configuration['api_key'],
      '#states' => [
        'required' => [
          ':input[name="configuration[asaas_pix][mode]"]' => [
            'value' => 'live',
          ],
        ],
      ],
    ];

    $form['api_key_sandbox'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sandbox API key'),
      '#size' => 255,
      '#maxlength' => 255,
      '#default_value' => $this->configuration['api_key_sandbox'],
      '#states' => [
        'required' => [
          ':input[name="configuration[asaas_pix][mode]"]' => [
            'value' => 'test',
          ],
        ],
      ],
    ];

    $form['endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live endpoint'),
      '#default_value' => $this->configuration['endpoint'],
    ];

    $form['endpoint_sandbox'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Sandbox endpoint'),
      '#default_value' => $this->configuration['endpoint_sandbox'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $configuration = $form_state->getValue('configuration');
    $endpoint = $configuration['asaas_pix']['endpoint'];
    $endpoint_sandbox = $configuration['asaas_pix']['endpoint_sandbox'];

    if (!filter_var($endpoint, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('endpoint', $this->t('The live endpoint entered is not valid. Please provide a valid URL.'));
    }

    if (!filter_var($endpoint_sandbox, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('endpoint_sandbox', $this->t('The sandbox endpoint entered is not valid. Please provide a valid URL.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['api_key'] = $values['api_key'];
      $this->configuration['api_key_sandbox'] = $values['api_key_sandbox'];
      $this->configuration['endpoint'] = $values['endpoint'];
      $this->configuration['endpoint_sandbox'] = $values['endpoint_sandbox'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->assertPaymentState($payment, ['new']);

    // For PIX payments, we create the payment in Asaas and set it to pending
    // The webhook will update it to completed when paid
    $api_calls = \Drupal::service('commerce_asaas.api_calls');

    // Get or create customer
    $owner = $payment->getOrder()->getCustomer();
    $billing_profile = $payment->getOrder()->getBillingProfile();

    $customer_id = '';
    if ($owner && !$owner->isAnonymous()) {
      $remote_ids = $owner->get('commerce_remote_id');
      if (!$remote_ids->isEmpty()) {
        $customer_id = $remote_ids->getByProvider('commerce_asaas');
      }

      if (!$customer_id) {
        $customer_id = $api_calls->createCustomer($this->configuration, $owner, $billing_profile);
        if ($customer_id) {
          $remote_ids->setByProvider('commerce_asaas', $customer_id);
          $owner->save();
        }
      }
    }

    if (!$customer_id) {
      throw new \InvalidArgumentException('Customer ID is required for PIX payment processing.');
    }

    // Create payment in Asaas
    $remote_id = $api_calls->createPayment($payment, $this->configuration, $customer_id, $capture);

    if (!$remote_id) {
      throw new \Exception('Failed to create PIX payment in Asaas API.');
    }

    // Set payment to pending - webhook will update to completed when paid
    $payment->setState('pending');
    $payment->setRemoteId($remote_id);
    $payment->save();

    // Update order state to validation (this makes it appear in order listings)
    $order = $payment->getOrder();
    $order->set('state', 'validation');
    $order->save();
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payment = $payment_storage->create([
      'state' => 'completed',
      'amount' => $order->getBalance(),
      'payment_gateway' => $this->parentEntity->id(),
      'order_id' => $order->id(),
      'remote_id' => $request->query->get('payment_id'),
      'remote_state' => $request->query->get('status'),
    ]);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function onCancel(OrderInterface $order, Request $request) {
    $this->messenger()->addMessage($this->t('You have canceled checkout at @gateway but may resume the checkout process here when ready.', [
      '@gateway' => $this->getDisplayLabel(),
    ]));
  }

}
