<?php

namespace Drupal\commerce_asaas\PluginForm\OffsiteRedirect;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm as BasePaymentOffsiteForm;
use Drupal\commerce_asaas\ApiCalls;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the PIX payment form.
 */
class PixPaymentForm extends BasePaymentOffsiteForm implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The API calls service.
   *
   * @var \Drupal\commerce_asaas\ApiCalls
   */
  protected $apiCalls;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new self();
    $instance->apiCalls = $container->get('commerce_asaas.api_calls');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;
    $order = $payment->getOrder();
    $billing_profile = $order->getBillingProfile();

    // Create customer if needed.
    $owner = $order->getCustomer();
    if ($owner && !$owner->isAnonymous()) {
      $customer_id = $this->getRemoteCustomerId($owner);

      if (!$customer_id) {
        $customer_id = $this->apiCalls->createCustomer($payment->getPaymentGateway()->getPlugin()->getConfiguration(), $owner, $billing_profile);
        $this->setRemoteCustomerId($owner, $customer_id);
        $owner->save();
      }

      if ($customer_id) {
        // Create PIX payment.
        $remote_id = $this->apiCalls->createPayment($payment, $payment->getPaymentGateway()->getPlugin()->getConfiguration(), $customer_id, TRUE);

        if ($remote_id) {
          $payment->setRemoteId($remote_id);
          $payment->setState('pending');
          $payment->save();

          // Update order state to pending (this makes it appear in order listings)
          $order = $payment->getOrder();
          $order->getState()->applyTransitionById('place');
          $order->save();

          // Get PIX QR Code from Asaas API
          $pix_data = $this->apiCalls->getPixQrCode($payment->getPaymentGateway()->getPlugin()->getConfiguration(), $remote_id);

          \Drupal::logger('commerce_asaas')->info('PIX QR Code data: @data', ['@data' => json_encode($pix_data)]);

          if ($pix_data && isset($pix_data['encodedImage']) && isset($pix_data['payload'])) {
            $form['pix_info'] = [
              '#type' => 'container',
              '#attributes' => ['class' => ['pix-payment-info']],
            ];

            $form['pix_info']['qr_code'] = [
              '#type' => 'html_tag',
              '#tag' => 'img',
              '#attributes' => [
                'src' => 'data:image/png;base64,' . $pix_data['encodedImage'],
                'alt' => 'QR Code PIX',
                'class' => ['pix-qr-code'],
              ],
            ];

            $form['pix_info']['pix_code'] = [
              '#type' => 'textarea',
              '#title' => $this->t('PIX Code'),
              '#default_value' => $pix_data['payload'],
              '#attributes' => ['readonly' => 'readonly'],
              '#rows' => 3,
            ];

            $form['pix_info']['instructions'] = [
              '#markup' => '<p>' . $this->t('Scan the QR Code or copy the PIX code to make the payment.') . '</p>',
            ];

            $form['pix_info']['amount'] = [
              '#markup' => '<p><strong>' . $this->t('Amount: @amount', ['@amount' => $payment->getAmount()->__toString()]) . '</strong></p>',
            ];

            // Add JavaScript to check payment status.
            $form['#attached']['library'][] = 'commerce_asaas/pix_payment';
            $form['#attached']['drupalSettings']['commerce_asaas']['payment_id'] = $remote_id;
            $form['#attached']['drupalSettings']['commerce_asaas']['check_url'] = Url::fromRoute('commerce_asaas.check_payment', ['payment_id' => $remote_id])->toString();
          }
          else {
            // PIX data not available yet, show manual reload option
            $form['pix_info'] = [
              '#type' => 'container',
              '#attributes' => ['class' => ['pix-payment-info']],
            ];

            $form['pix_info']['loading'] = [
              '#markup' => '<div class="pix-loading-message">
                <p><strong>' . $this->t('PIX payment created successfully!') . '</strong></p>
                <p>' . $this->t('The PIX QR Code and payment code are being generated. This may take a few moments.') . '</p>
                <p>' . $this->t('Please refresh this page in a few seconds to see your PIX payment information.') . '</p>
                <p><a href="' . $payment_data['invoiceUrl'] . '" target="_blank" class="button">' . $this->t('Open PIX payment page') . '</a></p>
              </div>',
            ];

            // Add JavaScript to check payment status and reload
            $form['#attached']['library'][] = 'commerce_asaas/pix_payment';
            $form['#attached']['drupalSettings']['commerce_asaas']['payment_id'] = $remote_id;
            $form['#attached']['drupalSettings']['commerce_asaas']['check_url'] = Url::fromRoute('commerce_asaas.check_payment', ['payment_id' => $remote_id])->toString();
          }
        }
        else {
          throw new PaymentGatewayException('Failed to create PIX payment.');
        }
      }
    }

    return $form;
  }

  /**
   * Get remote customer ID from user.
   */
  protected function getRemoteCustomerId($owner) {
    $remote_id_field = $owner->get('commerce_remote_id');
    if (!$remote_id_field->isEmpty()) {
      $remote_ids = $remote_id_field->getValue();
      foreach ($remote_ids as $remote_id) {
        if (isset($remote_id['remote_id']) && $remote_id['remote_id'] &&
            isset($remote_id['remote_provider']) && $remote_id['remote_provider'] == 'asaas') {
          return $remote_id['remote_id'];
        }
      }
    }
    return NULL;
  }

  /**
   * Set remote customer ID for user.
   */
  protected function setRemoteCustomerId($owner, $customer_id) {
    $remote_id_field = $owner->get('commerce_remote_id');
    $remote_ids = $remote_id_field->getValue();

    // Remove existing Asaas remote ID.
    $remote_ids = array_filter($remote_ids, function ($remote_id) {
      return !isset($remote_id['remote_provider']) || $remote_id['remote_provider'] != 'asaas';
    });

    // Add new remote ID.
    $remote_ids[] = [
      'remote_id' => $customer_id,
      'remote_provider' => 'asaas',
    ];

    $owner->set('commerce_remote_id', $remote_ids);
  }

}
