<?php

namespace Drupal\commerce_asaas\Render;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use GuzzleHttp\Client;

/**
 * Class responsible for pre-render callbacks.
 */
class PreRender implements TrustedCallbackInterface {

  /**
   * Pre-render method.
   *
   * @param array $element
   *   The render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The modified render array.
   */
  public static function afterBuildCallback(array $element, FormStateInterface $form_state) {
    $element['postal_code']['#weight'] = 0.0059;
    $element['administrative_area']['#weight'] = 0.01;
    $element['locality']['#weight'] = 0.011;
    $element['address_line2']['#title_display'] = 'before';
    $element['address_line2']['#title'] = t('Complement');
    $element['given_name']['#title'] = t('Full name');
    $element['given_name']['#size'] = 60;
    unset($element['family_name']);

    $element['postal_code']['#attached']['library'][] = 'core/internal.jquery.form';
    $element['postal_code']['#attached']['library'][] = 'core/drupal.ajax';
    $element['postal_code']['#ajax_processed'] = TRUE;
    $element['postal_code']['#ajax'] = [
      'callback' => [PreRender::class, 'ajaxConsultZip'],
      'event' => 'change',
      'progress' => [
        'type' => 'throbber',
        'message' => t('Consulting postal code...'),
      ],
    ];

    return $element;
  }

  /**
   * Call the function that consumes the web service.
   *
   * @param array $form
   *   A form that will be modified.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The values of the form.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The modified AjaxResponse.
   */
  public static function ajaxConsultZip(array &$form, FormStateInterface $form_state) {
    \Drupal::logger('Ajax')->info('Success');
    $ajax_response = new AjaxResponse();

    $parent_keys = $form_state->getTriggeringElement()['#array_parents'];
    $delta = $parent_keys[2];
    $parent_field_name = $parent_keys[0];
    $form_state_values = $form_state->getValues();

    // Retrieve the postal code based on parent keys.
    $elements = self::createArray($parent_keys, $form_state_values);
    $postal_code = NestedArray::getValue($elements, $parent_keys);
    $postal_code = preg_replace('/[^0-9\-]/', '', $postal_code);
    // Call the web service function and get the new values.
    $zip_code_response = self::consultZip($postal_code);

    if (!isset($zip_code_response->erro)) {
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[$delta][postal_code]\"]", 'val', [$zip_code_response->cep]));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[$delta][thoroughfare]\"]", 'val', [$zip_code_response->logradouro]));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[$delta][number]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[$delta][street_complement]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[$delta][neighborhood]\"]", 'val', [$zip_code_response->bairro]));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[$delta][city]\"]", 'val', [$zip_code_response->localidade]));
      $ajax_response->addCommand(new InvokeCommand("select[name=\"{$parent_field_name}[$delta][state]\"]", 'val', [$zip_code_response->uf]));
      $ajax_response->addCommand(new InvokeCommand("select[name=\"{$parent_field_name}[$delta][state]\"]", 'change'));
    }
    else {
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[0][postal_code]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[0][thoroughfare]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[0][number]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[0][street_complement]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[0][neighborhood]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[0][city]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("input[name=\"{$parent_field_name}[0][state]\"]", 'val', ['']));
      $ajax_response->addCommand(new InvokeCommand("select[name=\"{$parent_field_name}[0][state]\"]", 'change'));
    }

    return $ajax_response;
  }

  /**
   * Creates a deep array with the keys informed on $parent_keys.
   *
   * @param array $parent_keys
   *   Parent keys.
   * @param array $form_state_values
   *   Form state values.
   * @param array $created_array
   *   Created array.
   *
   * @return array
   *   Return array.
   */
  private static function createArray($parent_keys, $form_state_values, $created_array = []) {
    $key = array_shift($parent_keys);
    if (!isset($form_state_values[$key])) {
      $postal_code = reset($form_state_values);
      $created_array[$key][$parent_keys[0]] = $postal_code;
    }
    else {
      $created_array[$key] = self::createArray($parent_keys, $form_state_values[$key]);
    }
    return $created_array;
  }

  /**
   * Consume ViaCEP web service.
   *
   * @param int $zip
   *   The postal code to consult.
   *
   * @return mixed
   *   Street, neighborhood, city and state of the postal code.
   */
  public static function consultZip($zip) {
    try {
      $client = new Client();
      $response = $client->request('GET', "https://viacep.com.br/ws/{$zip}/json/", [
        'headers' => [
          'Accept' => 'application/json',
        ],
      ]);

      $output = $response->getBody()->getContents();
      return json_decode($output);
    }
    catch (\Exception $e) {
      \Drupal::logger('commerce_asaas')->error('Error consulting postal code: @error', ['@error' => $e->getMessage()]);
      return (object) ['erro' => TRUE];
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    // Register trusted callback methods.
    return ['afterBuildCallback', 'ajaxConsultZip'];
  }

}
