<?php

/**
 * @file
 * Token type definitions and replacements for Commerce Back in Stock.
 */

declare(strict_types=1);

use Drupal\Core\Datetime\DateFormat;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\user\Entity\User;
use Drupal\commerce_back_in_stock\StockSubscriptionStorageInterface;

/**
 * Implements hook_token_info().
 */
function commerce_back_in_stock_token_info(): array {
  $types['mynotify'] = [
    'name' => t('StockSubscription'),
    'description' => t('Tokens related to individual mynotify.'),
    'needs-data' => 'mynotify',
  ];
  $mynotify['id'] = [
    'name' => t('StockSubscription ID'),
    'description' => t('The unique ID of the mynotify.'),
  ];
  $mynotify['name'] = [
    'name' => t('Deprecated: StockSubscription Name'),
    'description' => t('Deprecated: StockSubscription Name.'),
  ];
  $mynotify['mail'] = [
    'name' => t('Email'),
    'description' => t('The email address of the StockSubscription.'),
  ];
  $mynotify['phone'] = [
    'name' => t('Phone'),
    'description' => t('The phone of the StockSubscription.'),
  ];
  $mynotify['text'] = [
    'name' => t('Text'),
    'description' => t('The text of the StockSubscription.'),
  ];
  $mynotify['edit-url'] = [
    'name' => t('Edit URL'),
    'description' => t('The URL of the StockSubscription edit page.'),
  ];
  $mynotify['created'] = [
    'name' => t('Created'),
    'description' => t('The date the StockSubscription was created.'),
    'type' => 'date',
  ];
  $mynotify['owner'] = [
    'name' => t('Owner'),
    'type' => 'user',
  ];
  $mynotify['product'] = [
    'name' => t('Product'),
    'type' => 'commerce_product',
  ];
  $mynotify['mails-not-notified'] = [
    'name' => t('Mails list not notified by product'),
    'description' => t('The list mails of the StockSubscription product_id.'),

  ];
  return [
    'types' => $types,
    'tokens' => ['mynotify' => $mynotify],
  ];
}

/**
 * Implements hook_tokens().
 *
 * Generates replacements for the 'mynotify' token type.
 */
function commerce_back_in_stock_tokens($type, array $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata): array {
  $token_service = \Drupal::token();
  $url_options = ['absolute' => TRUE];
  if (isset($options['langcode'])) {
    $url_options['language'] = \Drupal::languageManager()
      ->getLanguage($options['langcode']);
    $langcode = $options['langcode'];
  }
  else {
    $langcode = NULL;
  }
  $replacements = [];
  if ($type === 'mynotify' && !empty($data['mynotify'])) {
    /** @var \Drupal\commerce_back_in_stock\Entity\StockSubscription $mynotify */
    $mynotify = $data['mynotify'];
    foreach ($tokens as $name => $original) {
      switch ($name) {
        case 'id':
          // In the case of hook mynotify_presave id is not set yet.
          $replacements[$original] = $mynotify->id() ?: t('not yet assigned');
          break;

        case 'name':
          $replacements[$original] = $mynotify->getName();
          break;

        case 'mail':
          $replacements[$original] = $mynotify->getMail();
          break;

        case 'phone':
          $replacements[$original] = $mynotify->getPhone();
          break;

        case 'text':
          $replacements[$original] = $mynotify->getText();
          break;

        case 'edit-url':
          $replacements[$original] = $mynotify->id() ? $mynotify->url('edit-form', $url_options) : t('not yet assigned');
          break;

        case 'created':
          $date_format = DateFormat::load('medium');
          $bubbleable_metadata->addCacheableDependency($date_format);
          // In the case of mynotify_presave the created date may not yet be set.
          $replacements[$original] = $mynotify->getCreatedTime()
            ? \Drupal::service('date.formatter')
              ->format($mynotify->getCreatedTime(), 'medium', '', NULL, $langcode)
            : t('not yet created');
          break;

        case 'owner':
          $account = $mynotify->getOwner() ? $mynotify->getOwner() : User::load(0);
          $bubbleable_metadata->addCacheableDependency($account);
          $replacements[$original] = $account->label();
          break;

        case 'product':
          $product = $mynotify->getProduct() ? $mynotify->getProduct() : '';
          if ($product) {
            $replacements[$original] = $product->label();
          }
          else {
            $replacements[$original] = '';
          }
          break;

        case 'mails-not-notified':
          $srt = '';
          $product = $mynotify->getProduct() ? $mynotify->getProduct() : '';
          if ($product) {
            /** @var \Drupal\commerce_back_in_stock\StockSubscriptionStorageInterface $mynotify_storage */
            $mynotify_storage = \Drupal::entityTypeManager()
              ->getStorage('commerce_back_in_stock');
            $mynotifies = $mynotify_storage->loadPendingNotifications($product->id());
            if (!empty($mynotifies)) {
              $mails = [];
              foreach ($mynotifies as $mynotify) {
                $mail = $mynotify->getMail();
                $mails[$mail] = $mail;
              }
              $srt = implode(', ', $mails);
            }
          }
          $replacements[$original] = $srt;
          break;
      }

      if ($owner_tokens = $token_service->findWithPrefix($tokens, 'owner')) {
        $replacements += $token_service->generate('user', $owner_tokens, ['user' => $mynotify->getOwner()], $options, $bubbleable_metadata);
      }

      if ($product_tokens = $token_service->findWithPrefix($tokens, 'product')) {
        $replacements += $token_service->generate('commerce_product', $product_tokens, ['commerce_product' => $mynotify->getProduct()], $options, $bubbleable_metadata);
      }

    }

  }

  return $replacements;
}
