(function (Drupal) {
  'use strict';

  function formatUa(value) {
    // Allow only digits, then format to: +380 (XX) XXX-XX-XX
    let digits = String(value).replace(/\D/g, '');
    // Strip possible country code if user typed it; we add our own.
    if (digits.startsWith('380')) {
      digits = digits.slice(3);
    }
    if (digits.startsWith('0')) {
      digits = digits.slice(1);
    }
    // Limit to 9 subscriber digits for UA after country code.
    digits = digits.slice(0, 9);

    const a = digits.slice(0, 2);       // (XX)
    const b = digits.slice(2, 5);       // XXX
    const c = digits.slice(5, 7);       // XX
    const d = digits.slice(7, 9);       // XX

    let out = '+380';
    if (a.length) {
      out += ' (' + a;
      if (a.length === 2) {
        out += ')';
      }
    }
    if (b.length) {
      out += (a.length ? ' ' : ' (') + b;
      if (a.length < 2 && b.length && a.length < 2) {
        // Ensure closing paren if user jumped ahead (edge typing cases).
        if (a.length === 2) out += ') ';
      }
    }
    if (a.length === 2 && b.length && out.indexOf(')') === -1) {
      out = out.replace(' (' + a, ' (' + a + ')');
    }
    if (c.length) {
      // Ensure spacing before c.
      if (!/\)\s/.test(out)) {
        out = out.replace(')', ') ');
      }
      out += '-' + c;
    }
    if (d.length) {
      out += '-' + d;
    }
    return out;
  }

  Drupal.behaviors.commerceBackInStockPhone = {
    attach: function attach(context) {
      const inputs = context.querySelectorAll('[data-phone-mask]');
      inputs.forEach(function (el) {
        if (el.dataset.phoneMaskProcessed) {
          return;
        }
        el.dataset.phoneMaskProcessed = '1';

        const mode = el.getAttribute('data-phone-mask');
        if (mode === 'ua') {
          // Initialize with country code if empty.
          if (!el.value) {
            el.value = '+380 ';
          }
          const handler = function () {
            const caretEnd = el.selectionEnd;
            const formatted = formatUa(el.value);
            el.value = formatted;
            // Best-effort caret: move to end to keep UX simple.
            try { el.setSelectionRange(formatted.length, formatted.length); } catch (e) {}
          };
          el.addEventListener('input', handler);
          el.addEventListener('blur', handler);
          el.addEventListener('focus', function () {
            if (!el.value || el.value === '+380') {
              el.value = '+380 ';
            }
          });
        } else {
          // Default lightweight sanitizer.
          el.addEventListener('input', function () {
            const cleaned = el.value.replace(/[^0-9+()\-\s]/g, '');
            if (cleaned !== el.value) {
              el.value = cleaned;
            }
          });
        }
      });
    },
  };
})(Drupal);
