<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drush\Commands\DrushCommands;

/**
 * Drush commands for Commerce Back in Stock.
 */
class CommerceStockNotifyCommands extends DrushCommands {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected QueueFactory $queueFactory,
    protected QueueWorkerManagerInterface $queueWorkerManager,
  ) {
    parent::__construct();
  }

  /**
   * Run Commerce Back in Stock queues (customer and admin).
   *
   * @command commerce-back-in-stock:run
   * @aliases cbs:run
   */
  public function runQueues(): int {
    $queues = [
      'commerce_back_in_stock_customer',
      'commerce_back_in_stock_admin',
    ];
    $processed = 0;
    foreach ($queues as $queue_name) {
      $processed += $this->runQueue($queue_name);
    }
    $this->logger()->success(dt('Processed @count queued items.', ['@count' => $processed]));
    return self::EXIT_SUCCESS;
  }

  /**
   * Seed test subscriptions for a product.
   *
   * @param int $product_id
   *   Commerce product ID.
   * @param int $count
   *   How many subscriptions to create (default 5).
   *
   * @command commerce-back-in-stock:seed
   * @aliases cbs:seed
   */
  public function seed(int $product_id, int $count = 5): int {
    $storage = $this->entityTypeManager->getStorage('commerce_back_in_stock');
    $created = 0;
    for ($i = 0; $i < $count; $i++) {
      /**
     * @var \Drupal\commerce_back_in_stock\Entity\StockSubscription $entity
     */
      $entity = $storage->create(
            [
              'product_id' => $product_id,
              'name' => 'Test User ' . ($i + 1),
              'mail' => 'user' . ($i + 1) . '@example.com',
              'phone' => '',
              'text' => 'Seeded by Drush',
              'notified' => 0,
            ]
        );
      $entity->save();
      $created++;
    }
    $this->logger()->success(
          dt(
              'Created @count subscriptions for product @pid.',
              [
                '@count' => $created,
                '@pid' => $product_id,
              ]
          )
      );
    return self::EXIT_SUCCESS;
  }

  /**
   * Run a single queue until empty.
   */
  protected function runQueue(string $queue_name): int {
    $queue = $this->queueFactory->get($queue_name);
    $worker = $this->queueWorkerManager->createInstance($queue_name);
    $processed = 0;
    while ($item = $queue->claimItem()) {
      try {
        $worker->processItem($item->data);
        $queue->deleteItem($item);
        $processed++;
      }
      catch (\Throwable $e) {
        $this->logger()->error(dt('Queue @name failed: @msg', ['@name' => $queue_name, '@msg' => $e->getMessage()]));
        // Release item for retry depending on backend policy.
        $queue->releaseItem($item);
        break;
      }
    }
    return $processed;
  }

}
