<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\commerce_back_in_stock\Entity\StockSubscription;
use Drupal\commerce_product\Entity\Product;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Utility\Token;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Builds the subscription form modal for Commerce Stock Notify.
 */
class SubscriptionController extends ControllerBase {

  /**
   * Entity form builder service.
   *
   * @var \Drupal\Core\Entity\EntityFormBuilderInterface
   */
  protected $entityFormBuilder;

  /**
   * Config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * Constructs a new SubscriptionController instance.
   *
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entity_form_builder
   *   The entity form builder.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(EntityFormBuilderInterface $entity_form_builder, ConfigFactoryInterface $config_factory, Token $token) {
    $this->entityFormBuilder = $entity_form_builder;
    $this->configFactory = $config_factory;
    $this->token = $token;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
          $container->get('entity.form_builder'),
          $container->get('config.factory'),
          $container->get('token'),
      );
  }

  /**
   * {@inheritdoc}
   */
  public function viewPage(Product $commerce_product): array {
    $entity = StockSubscription::create();
    $entity->setProductId($commerce_product->id());
    $mynotify_form = $this->entityFormBuilder->getForm($entity, 'add_page');
    $noindex_nofollow = [
      '#tag' => 'meta',
      '#attributes' => [
        'name' => 'robots',
        'content' => 'noindex,nofollow',
      ],
    ];
    $mynotify_form['#attached']['html_head'][] = [
      $noindex_nofollow,
      'noindex_nofollow',
    ];
    return $mynotify_form;
  }

  /**
   * Page title callback for the subscription modal.
   *
   * @param \Drupal\commerce_product\Entity\Product $commerce_product
   *   The Commerce product.
   *
   * @return string
   *   The title string with tokens replaced.
   */
  public function viewPageTitle(Product $commerce_product): string {
    $settings_config = $this->configFactory->get('commerce_back_in_stock.settings');
    return (string) $this->token->replace($settings_config->get('popup.title'), ['commerce_product' => $commerce_product]);
  }

}
