<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Entity;

use Drupal\commerce_product\Entity\ProductInterface;
use Drupal\commerce_back_in_stock\CommerceStockNotifyInterface;
use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\UserInterface;

/**
 * Defines the StockSubscription entity.
 */
#[ContentEntityType(
    id: "commerce_back_in_stock",
    label: new TranslatableMarkup("Stock Subscription"),
    label_collection: new TranslatableMarkup("Stock Subscriptions"),
    label_singular: new TranslatableMarkup("stock subscription"),
    label_plural: new TranslatableMarkup("stock subscriptions"),
    entity_keys: [
      "id" => "id",
      "uuid" => "uuid",
      "langcode" => "langcode",
      "label" => "name",
      "owner" => "uid",
    ],
    handlers: [
      "storage" => "Drupal\commerce_back_in_stock\StockSubscriptionStorage",
      "storage_schema" => "Drupal\commerce_back_in_stock\StockSubscriptionSchema",
      "list_builder" => "Drupal\commerce_back_in_stock\StockSubscriptionListBuilder",
      "views_data" => "Drupal\views\EntityViewsData",
      "form" => [
        "add" => "Drupal\commerce_back_in_stock\Form\CommerceStockNotifyForm",
        "add_page" => "Drupal\commerce_back_in_stock\Form\CommerceStockNotifyForm",
        "edit" => "Drupal\commerce_back_in_stock\Form\CommerceStockNotifyForm",
        "delete" => "Drupal\commerce_back_in_stock\Form\CommerceStockNotifyDeleteForm",
      ],
      "access" => "Drupal\commerce_back_in_stock\CommerceStockAccessControlHandler",
    ],
    links: [
      "canonical" => "/admin/commerce/back-in-stock/{commerce_back_in_stock}/edit",
      "edit-form" => "/admin/commerce/back-in-stock/{commerce_back_in_stock}/edit",
      "delete-form" => "/admin/commerce/back-in-stock/{commerce_back_in_stock}/delete",
      "collection" => "/admin/commerce/back-in-stock",
    ],
    admin_permission: "administer commerce_back_in_stock entity",
    base_table: "commerce_back_in_stock",
    data_table: "commerce_back_in_stock_field_data",
    translatable: TRUE,
    field_ui_base_route: "entity.commerce_back_in_stock.collection"
)]
class StockSubscription extends ContentEntityBase implements CommerceStockNotifyInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'uid' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getProductId(): ?string {
    $value = $this->get('product_id')->target_id;
    return $value ? (string) $value : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setProductId($product_id): self {
    $this->set('product_id', $product_id);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getProduct(): ?ProductInterface {
    return $this->get('product_id')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getName(): string {
    return $this->get('name')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setName(string $name): self {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getMail(): string {
    return $this->get('mail')->value ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setMail(string $mail): self {
    $this->set('mail', $mail);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getPhone(): ?string {
    return $this->get('phone')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setPhone(string $phone): self {
    $this->set('phone', $phone);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isNotified(): bool {
    return (bool) $this->get('notified')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setNotified(bool $notified): self {
    $this->set('notified', $notified);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getText(): ?string {
    return $this->get('text')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setText(string $text): self {
    $this->set('text', $text);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('uid')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('uid', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('uid')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('uid', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    // Ensure standard fields exist; parent may not provide them for custom
    // content entities.
    if (!isset($fields['uid'])) {
      $fields['uid'] = BaseFieldDefinition::create('entity_reference')
        ->setLabel(t('Owner'))
        ->setDescription(t('The user ID of the subscription owner.'))
        ->setSetting('target_type', 'user')
        ->setSetting('handler', 'default')
        ->setDisplayConfigurable('form', TRUE)
        ->setDisplayConfigurable('view', TRUE);
    }

    if (!isset($fields['created'])) {
      $fields['created'] = BaseFieldDefinition::create('created')
        ->setLabel(t('Created'))
        ->setDescription(t('The time that the subscription was created.'))
        ->setDisplayConfigurable('form', TRUE)
        ->setDisplayConfigurable('view', TRUE);
    }

    if (!isset($fields['changed'])) {
      $fields['changed'] = BaseFieldDefinition::create('changed')
        ->setLabel(t('Changed'))
        ->setDescription(t('The time that the subscription was last edited.'));
    }

    // Provide the language field for translatable content entities.
    if ($entity_type->isTranslatable() && !isset($fields['langcode'])) {
      $fields['langcode'] = BaseFieldDefinition::create('language')
        ->setLabel(t('Language'))
        ->setDescription(t('The language of this subscription.'))
        ->setTranslatable(TRUE)
        ->setDisplayConfigurable('form', TRUE)
        ->setDisplayConfigurable('view', TRUE);
    }

    // Configure existing fields' display.
    $fields['uid']
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);
    $fields['created']
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // --- Кастомные поля ---
    $fields['product_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Product'))
      ->setSetting('target_type', 'commerce_product')
      ->setRequired(TRUE)
      ->setDisplayOptions(
              'view', [
                'label' => 'hidden',
                'type' => 'entity_reference_label',
                'weight' => 0,
              ]
          )
      ->setDisplayOptions(
              'form', [
                'type' => 'entity_reference_autocomplete',
                'weight' => 5,
                'settings' => [
                  'match_operator' => 'CONTAINS',
                  'size' => '60',
                  'placeholder' => '',
                ],
              ]
          )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions(
              'view', [
                'label' => 'inline',
                'type' => 'string',
                'weight' => 10,
              ]
          )
      ->setDisplayOptions(
              'form', [
                'type' => 'string_textfield',
                'weight' => 10,
              ]
          )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['mail'] = BaseFieldDefinition::create('email')
      ->setLabel(t('Email'))
      ->setRequired(TRUE)
      ->setDisplayOptions(
              'view', [
                'label' => 'inline',
                'type' => 'email_mailto',
                'weight' => 20,
              ]
          )
      ->setDisplayOptions(
              'form', [
                'type' => 'email_default',
                'weight' => 20,
              ]
          )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['phone'] = BaseFieldDefinition::create('telephone')
      ->setLabel(t('Phone'))
      ->setRequired(FALSE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions(
              'view', [
                'label' => 'inline',
                'type' => 'string',
                'weight' => 30,
              ]
          )
      ->setDisplayOptions(
              'form', [
                'type' => 'telephone_default',
                'weight' => 30,
              ]
          )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['text'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Message'))
      ->setRequired(FALSE)
      ->setDisplayOptions(
              'view', [
                'label' => 'above',
                'type' => 'basic_string',
                'weight' => 40,
              ]
          )
      ->setDisplayOptions(
              'form', [
                'type' => 'string_textarea',
                'weight' => 40,
                'settings' => ['rows' => 4],
              ]
          )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['notified'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Notified'))
      ->setDescription(t('A boolean indicating whether the customer has been notified.'))
      ->setDefaultValue(FALSE)
      ->setDisplayOptions(
              'form', [
                'type' => 'boolean_checkbox',
                'weight' => 50,
              ]
          )
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

}
