<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\EventSubscriber;

use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_back_in_stock\Service\NotificationManager;
use Drupal\Core\Entity\EntityEvents;
use Drupal\Core\Entity\Event\EntityCrudEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to entity update events to handle product availability changes.
 *
 * Note: We intentionally keep the logic simple and delegate the heavy work to
 * NotificationManager. The legacy hook implementation has been removed to
 * prevent duplicate processing.
 */
class ProductAvailabilitySubscriber implements EventSubscriberInterface {

  public function __construct(
    protected NotificationManager $notificationManager,
  ) {
  }

  /**
   * {@inheritdoc}
   *
   * Subscribes to entity update events.
   */
  public static function getSubscribedEvents(): array {
    // Use the event name string to avoid autoload issues in minimal test envs.
    return [
      'entity.update' => 'onEntityUpdate',
    ];
  }

  /**
   * React on entity updates.
   */
  public function onEntityUpdate(EntityCrudEvent $event): void {
    $entity = $event->getEntity();
    if ($entity instanceof Product) {
      // Let the manager decide whether the product is actually available and
      // enqueue work accordingly. This mirrors the previous hook behavior.
      $this->notificationManager->onProductUpdate($entity);
    }
  }

}
